import React, { memo, useEffect, useState } from 'react';
import { useDispatch } from 'react-redux';
import './application-event-source.component.less';
import { DotForm, DotInputText, DotCheckbox, DotButton, DotTypography, DotAutoComplete, DotDivider } from '@digital-ai/dot-components';
import { ENDPOINT_AUTHENTICATION_METHOD, EVENT_SOURCE_AUTH_OPTIONS, IMG_SRC } from '../../../constants';
import { IdAutocompleteOption, WebhookEndpoint } from '../../../external-deployment.types';
import { getFilterEventSourceSelector } from '../../../ducks/external-deployments.selectors';
import { useAppSelector } from '../../../../../../../../../../../core/xlr-ui/app/js/hooks';
import { folderExternalDeployments } from '../../../ducks/external-deployments.reducer';
import { Authentication } from '../../../../../../../../../../../core/xlr-ui/app/types';

export interface EventSourceFormProps {
    closeForm: () => void;
    eventSourceId: string;
}

const { loadFilterEventSource, saveEventSource } = folderExternalDeployments.actions;

const EventSourceFormComponent = (props: EventSourceFormProps) => {
    const { closeForm, eventSourceId } = props;

    const filterEventSource: WebhookEndpoint | undefined = useAppSelector(getFilterEventSourceSelector);

    const [title, setTitle] = useState<string>('');
    const [sourceEnabled, setSourceEnabled] = useState<boolean>(true);
    const [path, setPath] = useState<string | undefined>(undefined);
    const [authentication, setAuthentication] = useState<Authentication | undefined>(undefined);
    const [authToken, setAuthToken] = useState<string | undefined>(undefined);
    const [githubToken, setGithubToken] = useState<string | undefined>(undefined);
    const [jythonAuthScript, setJythonAuthScript] = useState<string | undefined>(undefined);
    const [requestRetention, setRequestRetention] = useState<number | undefined>(undefined);

    const dispatch = useDispatch();

    useEffect(() => {
        dispatch(loadFilterEventSource(eventSourceId));
    }, []);

    useEffect(() => {
        setTitle(filterEventSource ? filterEventSource.title : '');
        setSourceEnabled(filterEventSource ? filterEventSource.sourceEnabled : true);
        setPath(filterEventSource ? filterEventSource?.path : '');
        setAuthentication(filterEventSource ? filterEventSource.authentication : undefined);
        setRequestRetention(filterEventSource ? filterEventSource.requestRetention : undefined);
        setAuthToken(filterEventSource ? filterEventSource.authentication.webhookToken : '');
        setGithubToken(filterEventSource ? filterEventSource.authentication.githubSecret : '');
        setJythonAuthScript(filterEventSource ? filterEventSource.authentication.authenticationScript : '');
    }, [filterEventSource]);

    const onAuthMethodChange = (authMethodSelected: IdAutocompleteOption) => {
        if (authMethodSelected) {
            const auth: Authentication = {
                id: authentication ? authentication.id : null,
                type: authMethodSelected.id,
                githubSecret: githubToken,
                webhookToken: authToken,
                authenticationScript: jythonAuthScript,
            };
            setAuthentication(auth);
        } else {
            setAuthentication(undefined);
        }
    };

    const onSubmit = () => {
        const eventSource: WebhookEndpoint = {
            ...filterEventSource!,
            title,
            sourceEnabled,
            path,
            requestRetention,
            authentication: resolveFormAuthentication(authentication!),
        };

        dispatch(saveEventSource(eventSource));
        closeForm();
    };

    const resolveFormAuthentication = (auth: Authentication) => {
        const submittedAuth: Authentication = { id: auth.id, type: auth.type };
        if (auth.type === ENDPOINT_AUTHENTICATION_METHOD.tokenAuth) submittedAuth.webhookToken = authToken;
        else if (auth.type === ENDPOINT_AUTHENTICATION_METHOD.githubAuth || ENDPOINT_AUTHENTICATION_METHOD.githubJythonAuth)
            submittedAuth.githubSecret = githubToken;
        else if (auth.type === ENDPOINT_AUTHENTICATION_METHOD.jythonScriptAuth) submittedAuth.authenticationScript = jythonAuthScript;
        return submittedAuth;
    };

    return (
        <div className="event-source-form">
            <DotTypography variant="h2">
                <img alt="Webhook endpoint" src={IMG_SRC.webhookSource} />
                Endpoint webhook
            </DotTypography>
            <DotForm onSubmit={onSubmit}>
                <DotInputText
                    id="event-source-name"
                    label="Symbolic name for the connection"
                    name="eventSourceName"
                    onChange={(e) => setTitle(e.target.value)}
                    required
                    value={title}
                />
                <DotCheckbox checked={sourceEnabled} label="Enabled" name="enableEventSource" onChange={() => setSourceEnabled(!sourceEnabled)} />
                <DotInputText
                    helperText="Provide URl"
                    id="event-source-path"
                    label="Endpoint path"
                    name="endpointPath"
                    onChange={(e) => setPath(e.target.value)}
                    required
                    value={path}
                />
                <DotInputText
                    helperText="The maximum number of requests"
                    id="event-source-retention"
                    label="Request retention"
                    name="requestRetention"
                    onChange={(e) => setRequestRetention(+e.target.value)}
                    type="number"
                    value={'' + requestRetention}
                />
                <DotDivider />
                <div className="auth-header">
                    <DotTypography variant="h2">Authentication</DotTypography>
                </div>
                <DotAutoComplete
                    className="authentication-method"
                    freesolo={false}
                    inputId="authentication-method"
                    label="Authentication method"
                    multiple={false}
                    onChange={(e, v) => onAuthMethodChange(v as IdAutocompleteOption)}
                    options={EVENT_SOURCE_AUTH_OPTIONS}
                    required
                    value={EVENT_SOURCE_AUTH_OPTIONS.find(
                        (authOption) => authOption.id === (authentication ? authentication.type : ENDPOINT_AUTHENTICATION_METHOD.noAuth),
                    )}
                />
                {authentication?.type === ENDPOINT_AUTHENTICATION_METHOD.tokenAuth && (
                    <DotInputText
                        helperText="Secret token that will be passed in x-release-webhook-token request header"
                        id="event-source-token"
                        label="Secret Token"
                        name="token"
                        onChange={(e) => setAuthToken(e.target.value)}
                        type="password"
                        value={authentication ? authentication.webhookToken : ''}
                    />
                )}
                {authentication?.type === ENDPOINT_AUTHENTICATION_METHOD.githubAuth && (
                    <DotInputText
                        helperText="Secret token as defined on the Github Webhook configuration page"
                        id="event-source-webhook-secret"
                        label="Github Webhook Secret"
                        name="githubWebhookSecret"
                        onChange={(e) => setGithubToken(e.target.value)}
                        type="password"
                        value={authentication ? authentication.githubSecret : ''}
                    />
                )}
                {authentication?.type === ENDPOINT_AUTHENTICATION_METHOD.githubJythonAuth && (
                    <DotInputText
                        id="event-source-secret"
                        label="Github Secret"
                        name="githubSecret"
                        onChange={(e) => setGithubToken(e.target.value)}
                        type="password"
                        value={authentication ? authentication.githubSecret : ''}
                    />
                )}
                {authentication?.type === ENDPOINT_AUTHENTICATION_METHOD.jythonScriptAuth && (
                    <DotInputText
                        helperText="Jython script that will decide if an incoming request is considered authenticated.
                        This script should set the variable `authenticated` to either `True` or `False`.
                        The script can access the properties of the HTTP request by using the following variables:
                        * endpoint: this HTTP endpoint for Webhooks * config: the HTTP endpoint's authentication method * headers: the headers of the current HTTP request
                        * parameters: the URL parameters of the current HTTP request * payload: the request body of the current HTTP request"
                        id="event-source-jython-script"
                        label="Authentication Script"
                        multiline={true}
                        name="jythonScript"
                        onChange={(e) => setJythonAuthScript(e.target.value)}
                        size="medium"
                        value={authentication ? authentication.authenticationScript : ''}
                    />
                )}
                <div style={{ textAlign: 'right' }}>
                    <DotButton onClick={closeForm} type="text">
                        Cancel
                    </DotButton>
                    <DotButton disabled={!authentication || !title || !path} isSubmit={true}>
                        Save
                    </DotButton>
                </div>
            </DotForm>
        </div>
    );
};

export const EventSourceForm = memo(EventSourceFormComponent);
