import React, { memo, useEffect, useState } from 'react';
import { useDispatch } from 'react-redux';
import './application-create-server-component.less';
import { DotAlertBanner, DotAutoComplete, DotButton, DotDivider, DotForm, DotInputText, DotTypography } from '@digital-ai/dot-components';
import { ENDPOINT_TYPE, SERVER_AUTHENTICATION_METHOD, SERVER_AUTHENTICATION_OPTIONS, SupportedServerType } from '../../../constants';
import { IdAutocompleteOption, Server } from '../../../external-deployment.types';
import { Folder } from '../../../../../../../../../../../core/xlr-ui/app/types';
import { folderExternalDeployments } from '../../../ducks/external-deployments.reducer';
import { getServerIcon } from '../../../helper/utils';
import { externalApplicationWizard, getApplicationWizardState } from '../../../ducks/application-wizard.reducer';
import { useAppSelector } from '../../../../../../../../../../../core/xlr-ui/app/js/hooks';

export interface ApplicationCreateServerFormProp {
    closeForm: () => void;
    folder: Folder;
    serverType: SupportedServerType;
}

const { createServer } = folderExternalDeployments.actions;
const { setConnectionTested, testConnection } = externalApplicationWizard.actions;

const CreateServerFormComponent = (props: ApplicationCreateServerFormProp) => {
    const { folder, serverType, closeForm } = props;
    const {
        connectionTested,
        testConnectionStatus: { errorText, success },
    } = useAppSelector(getApplicationWizardState);

    const [authMethod, setAuthMethod] = useState<string>(SERVER_AUTHENTICATION_METHOD.basic);
    const [authToken, setAuthToken] = useState<string | undefined>();
    const [title, setTitle] = useState<string>('');
    const [titleError, setTitleError] = useState<boolean>(false);
    const [titleTouched, setTitleTouched] = useState<boolean>(false);
    const [url, setUrl] = useState<string>('');
    const [urlError, setUrlError] = useState<boolean>(false);
    const [urlTouched, setUrlTouched] = useState<boolean>(false);
    const [username, setUsername] = useState<string | undefined>();
    const [password, setPassword] = useState<string | undefined>();
    const [domain, setDomain] = useState<string | undefined>();
    const [accessTokenUrl, setAccessTokenUrl] = useState<string | undefined>();
    const [clientId, setClientId] = useState<string | undefined>();
    const [clientSecret, setClientSecret] = useState<string | undefined>();
    const [scope, setScope] = useState<string | undefined>();
    const [proxyHost, setProxyHost] = useState<string | undefined>();
    const [proxyPort, setProxyPort] = useState<number | undefined>();
    const [proxyUsername, setProxyUsername] = useState<string | undefined>();
    const [proxyPassword, setProxyPassword] = useState<string | undefined>();
    const [proxyDomain, setProxyDomain] = useState<string | undefined>();
    const [saveEnabled, setSaveEnabled] = useState<boolean | undefined>();

    const dispatch = useDispatch();

    useEffect(() => {
        setSaveEnabled(title.length !== 0 && url.length !== 0);
        setTitleError(title.length === 0 && titleTouched);
        setUrlError(url.length === 0 && urlTouched);
    }, [title, url, titleTouched, urlTouched]);

    const onSubmit = () => {
        const ci: Server = {
            accessTokenUrl,
            authenticationMethod: authMethod,
            authToken,
            clientId,
            clientSecret,
            domain,
            folderId: folder.id,
            id: null,
            password,
            proxyDomain,
            proxyHost,
            proxyPassword,
            proxyPort,
            proxyUsername,
            scope,
            title,
            type: serverType,
            url,
            username,
        };
        dispatch(createServer(ci));
        closeForm();
    };

    const onAuthMethodChange = (authMethodSelected: IdAutocompleteOption) => {
        if (authMethodSelected) {
            setAuthMethod(authMethodSelected.id);
        } else {
            setAuthMethod('');
        }
    };

    const onTestConnection = async () => {
        const requestBody = {
            id: null,
            properties: {
                accessTokenUrl,
                authenticationMethod: authMethod,
                authToken,
                clientId,
                clientSecret,
                domain,
                folderId: folder.id,
                id: null,
                password,
                proxyHost,
                proxyPort,
                proxyUsername,
                proxyPassword,
                proxyDomain,
                scope,
                url,
                username,
                title,
                type: serverType,
            },
            title,
            type: serverType,
        };
        dispatch(testConnection(requestBody));
    };

    const generateTestMessage = (serverTypeName: string) => {
        return (
            <DotAlertBanner
                action={
                    <DotButton onClick={() => dispatch(setConnectionTested(false))} type="text">
                        Dismiss
                    </DotButton>
                }
                className="create-server-banner"
                severity={success ? 'success' : 'error'}
            >
                {success ? (
                    <div className="create-server-test-message">{serverTypeName} is available.</div>
                ) : (
                    <div className="create-server-test-message">
                        Can&apos;t connect to {serverTypeName}.<br />
                        {errorText}
                    </div>
                )}
            </DotAlertBanner>
        );
    };

    const generateAuthFields = () => {
        if (serverType === ENDPOINT_TYPE.argoCd) {
            return (
                <>
                    {generateBasicAuth()}
                    <DotInputText
                        id="create-server-auth-token"
                        label="Auth Token"
                        name="create-server-auth-token"
                        onChange={(e) => setAuthToken(e.target.value)}
                        type="password"
                    />
                </>
            );
        } else {
            return (
                <>
                    {serverType === ENDPOINT_TYPE.deploy && (
                        <DotAutoComplete
                            className="authentication-method"
                            defaultValue={SERVER_AUTHENTICATION_METHOD.basic}
                            freesolo={false}
                            inputId="authentication-method"
                            label="Authentication methods"
                            multiple={false}
                            onChange={(e, v) => onAuthMethodChange(v as IdAutocompleteOption)}
                            options={SERVER_AUTHENTICATION_OPTIONS}
                            required
                        />
                    )}
                    {authMethod !== SERVER_AUTHENTICATION_METHOD.none && (
                        <>
                            {generateBasicAuth()}
                            {authMethod === SERVER_AUTHENTICATION_METHOD.ntlm && (
                                <DotInputText
                                    helperText="Domain to be used for NTLM authentication"
                                    id="create-server-domain"
                                    label="Domain"
                                    name="create-server-domain"
                                    onChange={(e) => setDomain(e.target.value)}
                                />
                            )}
                            {authMethod === SERVER_AUTHENTICATION_METHOD.oauth2 && (
                                <>
                                    <DotInputText
                                        helperText="The endpoint of the authentication server"
                                        id="create-server-access-token-url"
                                        label="Access Token URL"
                                        name="create-server-access-token-url"
                                        onChange={(e) => setAccessTokenUrl(e.target.value)}
                                    />
                                    <DotInputText
                                        helperText="The client identifier issued to the client during the Application registration process"
                                        id="create-server-client-id"
                                        label="Client ID"
                                        name="create-server-client-id"
                                        onChange={(e) => setClientId(e.target.value)}
                                    />
                                    <DotInputText
                                        helperText="The client secret issued to the client during the Application registration process"
                                        id="create-server-client-secret"
                                        label="Client Secret"
                                        name="create-server-client-secret"
                                        onChange={(e) => setClientSecret(e.target.value)}
                                        type="password"
                                    />
                                    <DotInputText
                                        helperText="The scope of the access request. It may have multiple space-delimited values"
                                        id="create-server-client-id"
                                        label="Scope"
                                        name="create-server-client-id"
                                        onChange={(e) => setScope(e.target.value)}
                                    />
                                </>
                            )}
                        </>
                    )}
                </>
            );
        }
    };

    const generateBasicAuth = () => {
        return (
            <>
                <DotDivider />
                <DotTypography variant="h2">Authentication</DotTypography>
                <DotInputText
                    helperText="Login user ID on the server"
                    id="create-server-username"
                    label="Username"
                    name="create-server-username"
                    onChange={(e) => setUsername(e.target.value)}
                />
                <DotInputText
                    helperText="Login user password on the server"
                    id="create-server-password"
                    label="Password"
                    name="create-server-password"
                    onChange={(e) => setPassword(e.target.value)}
                    type="password"
                />
            </>
        );
    };

    return (
        <>
            {connectionTested && generateTestMessage(serverType === ENDPOINT_TYPE.deploy ? 'Digital.ai Deploy Server' : 'ArgoCD Server')}

            <div className="create-server">
                <div className="create-server-header">
                    <div className="create-server-header-img">
                        <img alt={serverType} src={getServerIcon(serverType)} />
                    </div>
                    <DotTypography variant="h1">New {serverType === ENDPOINT_TYPE.deploy ? 'Deploy' : 'Argo'} server</DotTypography>
                </div>

                <DotForm onSubmit={onSubmit}>
                    <div className="create-server-inputs">
                        <DotInputText
                            error={titleError}
                            helperText={titleError ? 'Title is mandatory' : 'Symbolic name for the configuration'}
                            id="create-server-title"
                            label="Title"
                            name="create-server-title"
                            onBlur={() => setTitleTouched(true)}
                            onChange={(e) => setTitle(e.target.value)}
                            required
                        />
                        <DotInputText
                            error={urlError}
                            helperText={urlError ? 'Url is mandatory' : 'Address where the server can be reached'}
                            id="create-server-url"
                            label="URL"
                            name="create-server-url"
                            onBlur={() => setUrlTouched(true)}
                            onChange={(e) => setUrl(e.target.value)}
                            required
                        />

                        {generateAuthFields()}

                        <DotDivider />
                        <DotTypography variant="h2">Proxy</DotTypography>
                        <DotInputText
                            helperText="HTTP proxy host"
                            id="create-server-proxy-host"
                            label="Proxy Host"
                            name="create-server-proxy-host"
                            onChange={(e) => setProxyHost(e.target.value)}
                        />
                        <DotInputText
                            helperText="HTTP proxy port"
                            id="create-server-proxy-port"
                            label="Proxy Port"
                            name="create-server-proxy-port"
                            onChange={(e) => setProxyPort(+e.target.value)}
                            type="number"
                        />
                        <DotInputText
                            helperText="HTTP proxy username"
                            id="create-server-proxy-username"
                            label="Proxy Username"
                            name="create-server-proxy-username"
                            onChange={(e) => setProxyUsername(e.target.value)}
                        />
                        <DotInputText
                            helperText="HTTP proxy password"
                            id="create-server-proxy-password"
                            label="Proxy Password"
                            name="create-server-proxy-password"
                            onChange={(e) => setProxyPassword(e.target.value)}
                            type="password"
                        />
                        <DotInputText
                            helperText="Domain to be used for NTLM proxy authentication"
                            id="create-server-proxy-domain"
                            label="Proxy Domain"
                            name="create-server-proxy-domain"
                            onChange={(e) => setProxyDomain(e.target.value)}
                        />
                    </div>

                    <div className="create-server-buttons-wrapper">
                        <div className="create-server-buttons">
                            <DotButton onClick={closeForm} type="text">
                                Cancel
                            </DotButton>
                            <DotButton data-testid="test-connection-btn" disabled={!saveEnabled} onClick={onTestConnection} type="outlined">
                                Test
                            </DotButton>
                            <DotButton data-testid="save-server-btn" disabled={!saveEnabled} isSubmit={true}>
                                Save
                            </DotButton>
                        </div>
                    </div>
                </DotForm>
            </div>
        </>
    );
};

export const CreateServerForm = memo(CreateServerFormComponent);
