package com.xebialabs.xlrelease.status.sse.service

import com.fasterxml.jackson.databind.ObjectMapper
import com.fasterxml.jackson.datatype.joda.JodaModule
import com.fasterxml.jackson.module.scala.DefaultScalaModule
import com.xebialabs.xlrelease.status.webhook.events._
import grizzled.slf4j.Logging
import org.springframework.stereotype.Service

import java.io.EOFException
import java.util.ConcurrentModificationException
import java.util.concurrent.{CompletableFuture, CopyOnWriteArrayList}
import javax.ws.rs.core.MediaType
import javax.ws.rs.sse.{OutboundSseEvent, Sse, SseEventSink}

trait ServerSentEventsService {
  def add(sseEventSink: SseEventSink, sse: Sse): Unit

  def send(id: String, event: ExternalDeploymentEvent): Unit
}

@Service
class ServerSentEventsServiceImpl extends ServerSentEventsService with Logging {

  private val sinks = new CopyOnWriteArrayList[SseEventSink]()
  private var sse :Sse = _
  private val mapper = new ObjectMapper()
  mapper.registerModule(DefaultScalaModule)
  mapper.registerModule(new JodaModule)

  override def add(sseEventSink: SseEventSink, sse: Sse): Unit = {
    this.sse = sse
    sinks.add(sseEventSink)

    logger.info(s"New emitter $sseEventSink has been added")

  }

  def send(id: String, event: ExternalDeploymentEvent): Unit = {
    val sseEvent = buildSseEvent(EndpointExternalDeploymentEvent(id, event))

    sinks.forEach(emitter => {
      CompletableFuture.runAsync(() => {
        emitter.send(sseEvent)
      }).whenComplete((_, ex) => {
        Option(ex.getCause).map {
          case _: EOFException | _: ConcurrentModificationException =>
            if (sinks.remove(emitter)) {
              emitter.close()
              logger.info(s"Client connection for emitter $emitter was terminated, removing emitter $emitter")
            }
          case ex: Throwable =>
            if (sinks.remove(emitter)) {
              emitter.close()
              logger.error(s"Emitter $emitter failed: $ex")
            }
          case _ => // did not happen
        }
      })
    })
  }

  private def buildSseEvent(event: EndpointExternalDeploymentEvent): OutboundSseEvent = {

    var name :String = ""

    event.state match {
      case e: CreateStatusEvent if isApplicationEvent(e) => name = "application-created"
      case e: DeleteStatusEvent if isApplicationEvent(e) => name = "application-deleted"
      case _: CreateStatusEvent => name = "application-package-created"
      case _: DeleteStatusEvent => name = "application-package-deleted"
      case _: UpdateStatusEvent => name = "application-changed"
      case _ => // ignore
    }

    sse.newEventBuilder()
          .name(name)
          .mediaType(MediaType.APPLICATION_JSON_TYPE)
          .data(mapper.writeValueAsString(event))
          .reconnectDelay(3000)
          .build()
  }

  private def isApplicationEvent(event: BaseExternalDeploymentEvent) = {
    Option(event.destination).isDefined || Option(event.namespace).isDefined
  }
}

sealed trait ServerSentEvent {
  val endpointId: String
}

case class EndpointExternalDeploymentEvent(override val endpointId: String, state: ExternalDeploymentEvent) extends ServerSentEvent
