import React from 'react';
import { ReactWrapper } from 'enzyme';
import { DotAutoComplete, DotButton, DotCheckbox, DotForm, DotInputText } from '@digital-ai/dot-components';
import { mountWithStoreAndTheme } from '../../../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import { mockDeployWebhookEventSource } from '../../../__mocks__/external-deployments.mocks';
import { folderExternalDeployments, FolderExternalDeploymentsState, initialState } from '../../../ducks/external-deployments.reducer';
import { ENDPOINT_AUTHENTICATION_METHOD, EVENT_SOURCE_AUTH_OPTIONS, IMG_SRC } from '../../../constants';
import { EventSourceForm } from './application-event-source.component';
import { IdAutocompleteOption } from '../../../external-deployment.types';

const { loadFilterEventSource, saveEventSource } = folderExternalDeployments.actions;
const dispatch = jest.fn();

describe('EventSourceForm component', () => {
    let wrapper: ReactWrapper;
    const closeForm = jest.fn();
    const defaultProps = {
        closeForm,
        eventSourceId: mockDeployWebhookEventSource.id || '',
    };
    const defaultState = {
        ...initialState,
    };
    const mountComponent = (state: FolderExternalDeploymentsState = defaultState) => {
        wrapper = mountWithStoreAndTheme(<EventSourceForm {...defaultProps} />, dispatch, { folderExternalDeployments: state });
        expect(dispatch).toBeCalledWith(loadFilterEventSource(mockDeployWebhookEventSource.id || ''));
    };

    beforeEach(() => {
        mountComponent();
    });

    afterEach(() => {
        wrapper.unmount();
    });

    const searchDotCard = (selector: string) => wrapper.find(`.event-source-form ${selector}`);

    it('should show the header', () => {
        expect(searchDotCard('.dot-typography img').at(0).props().src).toStrictEqual(IMG_SRC.webhookSource);
        expect(searchDotCard('.dot-typography').at(0).text()).toStrictEqual('Endpoint webhook');
    });

    it('should show form fields', () => {
        expect(searchDotCard('[name="eventSourceName"] input')).toHaveLength(1);
        expect(searchDotCard('[name="enableEventSource"] input')).toHaveLength(1);
        expect(searchDotCard('[name="endpointPath"] input')).toHaveLength(1);
        expect(searchDotCard('[name="requestRetention"] input')).toHaveLength(1);
        expect(searchDotCard('[name="Authentication method"] input')).toHaveLength(1);
    });

    it('should show buttons', () => {
        expect(searchDotCard('.dot-button button').at(0).text()).toStrictEqual('Cancel');
        expect(searchDotCard('.dot-button button').at(1).text()).toStrictEqual('Save');
    });

    it('should handle submit with undefined filterEventSource', () => {
        wrapper.find(DotForm).invoke('onSubmit')?.('' as never);
        wrapper.update();
        expect(closeForm).not.toHaveBeenCalled();
    });

    it('should change auth method to null', () => {
        mountComponent({ ...defaultState, filterEventSource: mockDeployWebhookEventSource });
        wrapper.find(DotAutoComplete).invoke('onChange')?.('' as never, null, '');
        wrapper.find(DotInputText).at(0).invoke('onChange')?.({ target: { value: 'title' } } as never);
        wrapper.find(DotInputText).at(1).invoke('onChange')?.({ target: { value: 'path' } } as never);
        wrapper.update();
        expect(wrapper.find(DotButton).at(1).props().disabled).toBeTruthy();
    });

    const shouldChangeAuth = (eventSourceAuth: IdAutocompleteOption) => {
        wrapper.find(DotAutoComplete).invoke('onChange')?.('' as never, eventSourceAuth, '');
        wrapper.update();
        wrapper.find(DotInputText).at(0).invoke('onChange')?.({ target: { value: 'title' } } as never);
        wrapper.find(DotInputText).at(1).invoke('onChange')?.({ target: { value: 'path' } } as never);
        if (wrapper.find(DotInputText).at(3).length > 0) {
            wrapper.find(DotInputText).at(3)?.invoke('onChange')?.({ target: { value: 'auth value' } } as never);
        }
        wrapper.update();
        expect(wrapper.find(DotButton).at(1).props().disabled).toBeFalsy();
    };

    const shouldSubmitWithAuth = (eventSourceAuth: IdAutocompleteOption, sourceEnabled = true, requestRetentionParam: number | undefined = undefined) => {
        mountComponent({ ...defaultState, filterEventSource: { ...mockDeployWebhookEventSource } });
        if (!sourceEnabled) {
            wrapper.find(DotCheckbox).invoke('onChange')?.('' as never, 'false');
        }
        const requestRetention = requestRetentionParam ? requestRetentionParam : mockDeployWebhookEventSource.requestRetention;
        if (requestRetentionParam) {
            wrapper.find(DotInputText).at(2).invoke('onChange')?.({ target: { value: requestRetentionParam } } as never);
        }
        shouldChangeAuth(eventSourceAuth);
        wrapper.find(DotForm).invoke('onSubmit')?.('' as never);
        wrapper.update();
        const getAuthParamName = (): string | undefined => {
            if (eventSourceAuth.id === ENDPOINT_AUTHENTICATION_METHOD.tokenAuth) return 'webhookToken';
            else if (eventSourceAuth.id === ENDPOINT_AUTHENTICATION_METHOD.githubAuth || eventSourceAuth.id === ENDPOINT_AUTHENTICATION_METHOD.githubJythonAuth)
                return 'githubSecret';
            else if (eventSourceAuth.id === ENDPOINT_AUTHENTICATION_METHOD.jythonScriptAuth) return 'authenticationScript';
        };

        const authParamName = getAuthParamName();
        expect(dispatch).toHaveBeenCalledWith(
            saveEventSource({
                ...mockDeployWebhookEventSource,
                authentication: {
                    id: mockDeployWebhookEventSource.authentication.id,
                    type: eventSourceAuth.id,
                    [authParamName || 'githubSecret']: authParamName ? 'auth value' : undefined,
                },
                path: 'path',
                requestRetention,
                sourceEnabled,
                title: 'title',
            }),
        );
        expect(closeForm).toHaveBeenCalled();
    };
    it('should submit with no auth method', () => {
        shouldSubmitWithAuth(EVENT_SOURCE_AUTH_OPTIONS[0]);
    });
    it('should submit with token auth method', () => {
        shouldSubmitWithAuth(EVENT_SOURCE_AUTH_OPTIONS[1]);
    });
    it('should submit with giithub auth method', () => {
        shouldSubmitWithAuth(EVENT_SOURCE_AUTH_OPTIONS[2]);
    });
    it('should submit with github jython auth method', () => {
        shouldSubmitWithAuth(EVENT_SOURCE_AUTH_OPTIONS[3]);
    });
    it('should submit with script jython auth method', () => {
        shouldSubmitWithAuth(EVENT_SOURCE_AUTH_OPTIONS[4], false, 2);
    });
    it('should render token auth method', () => {
        shouldChangeAuth(EVENT_SOURCE_AUTH_OPTIONS[1]);
    });
    it('should render github auth method', () => {
        shouldChangeAuth(EVENT_SOURCE_AUTH_OPTIONS[2]);
    });
    it('should render github jython auth method', () => {
        shouldChangeAuth(EVENT_SOURCE_AUTH_OPTIONS[3]);
    });
    it('should render script jython auth method', () => {
        shouldChangeAuth(EVENT_SOURCE_AUTH_OPTIONS[4]);
    });
});
