package com.xebialabs.xlrelease.versioning.templates

import com.xebialabs.deployit.plumbing.export.ImportTemplateConfigurationItemReaderWriter
import com.xebialabs.deployit.plumbing.serialization.ResolutionContext
import com.xebialabs.xlrelease.domain.Release
import com.xebialabs.xlrelease.export.TemplateImportUpgrader
import com.xebialabs.xlrelease.repository.sql.SqlRepositoryAdapter
import com.xebialabs.xlrelease.repository.sql.persistence.CompressionSupport
import com.xebialabs.xlrelease.repository.{CiCloneHelper, Ids}
import com.xebialabs.xlrelease.utils.CiHelper
import org.codehaus.jettison.json.JSONObject
import org.jboss.resteasy.core.Headers
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import java.io.{ByteArrayInputStream, ByteArrayOutputStream}
import java.lang.annotation.Annotation
import java.nio.charset.{Charset, StandardCharsets}
import javax.ws.rs.core.MediaType

@Component
class ReleaseVersioningSerialization @Autowired()(configurationItemReaderWriter: ImportTemplateConfigurationItemReaderWriter,
                                                  repository: SqlRepositoryAdapter,
                                                  templateImportUpgrader: TemplateImportUpgrader) extends CompressionSupport {

  def toBytes(template: Release): Array[Byte] = {
    compress(toJson(template).getBytes(StandardCharsets.UTF_8))
  }

  def fromBytes(contentBytes: Array[Byte], version: String, context: ResolutionContext): Option[Release] = {
    val templateJsonBytes = decompress(contentBytes)
    val templateJson = new JSONObject(new String(templateJsonBytes, StandardCharsets.UTF_8))
    templateImportUpgrader.upgrade(templateJson, version)

    configurationItemReaderWriter.readFrom(context,
      MediaType.APPLICATION_JSON_TYPE,
      new ByteArrayInputStream(templateJson.toString.getBytes(Charset.defaultCharset()))
    ) match {
      case ci if ci.isInstanceOf[Release] => Some(ci.asInstanceOf[Release])
      case _ => None
    }
  }

  private def toJson(originalTemplate: Release): String = {
    val templateOut = new ByteArrayOutputStream()
    val template = CiCloneHelper.cloneCi(originalTemplate)
    CiHelper.rewriteWithNewId(template, Ids.getName(template.getId))
    configurationItemReaderWriter.writeTo(template,
      classOf[Release],
      classOf[Release],
      Array[Annotation](),
      MediaType.APPLICATION_JSON_TYPE,
      new Headers[AnyRef],
      templateOut
    )
    // configurationItemReaderWriter will use default charset so always make a correct string even if file.encoding is specified
    new String(templateOut.toByteArray, Charset.defaultCharset())
  }

}
