package com.xebialabs.xlrelease.versioning.templates

import com.xebialabs.deployit.checks.Checks.checkArgument
import com.xebialabs.deployit.plumbing.serialization.ResolutionContext
import com.xebialabs.deployit.security.Permissions.getAuthenticatedUserName
import com.xebialabs.xlrelease.actors.ReleaseActorService
import com.xebialabs.xlrelease.domain.Release
import com.xebialabs.xlrelease.dsl.service.DslExporter
import com.xebialabs.xlrelease.param.IdParam
import com.xebialabs.xlrelease.repository.{ConfigurationRepository, Ids, Page}
import com.xebialabs.xlrelease.scm.connector.{ScmConnectorService, ScmException}
import com.xebialabs.xlrelease.security.PermissionChecker
import com.xebialabs.xlrelease.security.XLReleasePermissions.{CREATE_RELEASE_FROM_TEMPLATE, EDIT_TEMPLATE}
import com.xebialabs.xlrelease.service.ReleaseService
import com.xebialabs.xlrelease.versioning.scm.ScmTemplateStorageService
import com.xebialabs.xlrelease.versioning.templates.actors.extensions.TemplateVersionExtensionActorMessages.{NewTemplateVersion, RestoreRevision}
import com.xebialabs.xlrelease.versioning.templates.configuration.domain.TemplateVersioningSettings
import com.xebialabs.xlrelease.versioning.templates.configuration.domain.TemplateVersioningSettings.TEMPLATE_VERSIONING_ID
import com.xebialabs.xlrelease.versioning.templates.repository.TemplateRevisionRepository
import com.xebialabs.xlrelease.versioning.templates.repository.persistence.data.TemplateRevision
import com.xebialabs.xlrelease.versioning.templates.views.{TemplateContentAndPrevious, VersionView}
import grizzled.slf4j.Logging
import org.apache.http.HttpStatus
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Controller

import java.util.{List => JList}
import javax.ws.rs.core.{GenericEntity, MediaType, Response}
import javax.ws.rs._
import scala.jdk.CollectionConverters._
import scala.util.Try


@Path("/versioning")
@Produces(Array(MediaType.APPLICATION_JSON))
@Consumes(Array(MediaType.APPLICATION_JSON))
@Controller
class TemplateVersioningResource @Autowired()(dslExporter: DslExporter,
                                              templateRevisionRepository: TemplateRevisionRepository,
                                              templateStorageService: TemplateStorageService,
                                              releaseActorService: ReleaseActorService,
                                              permissionChecker: PermissionChecker,
                                              releaseService: ReleaseService,
                                              configurationRepository: ConfigurationRepository,
                                              templateStorageListener: ScmTemplateStorageService,
                                              scmConnectorService: ScmConnectorService) extends Logging {

  @GET
  @Path("/settings")
  def getTemplateVersionoingSettings: TemplateVersioningSettings = {
    configurationRepository.read(TEMPLATE_VERSIONING_ID)
  }

  @GET
  @Path("{templateId:.*Release[^/-]*}/connectionInfo")
  def connectionInfo(@PathParam("templateId") @IdParam templateId: String): Response = {
    val folderId = Ids.findFolderId(templateId)
    val scmConfiguration = scmConnectorService.findScmConnectorConfig(folderId)
    Response.ok(scmConfiguration).build()
  }

  @GET
  @Path("/{templateId:.*Release[^/-]*}/revisions")
  def getRevisions(@PathParam("templateId") @IdParam templateId: String): Response = {
    permissionChecker.checkView(templateId)
    val templateRevisions = templateRevisionRepository.find(templateId, Page(0, Int.MaxValue, 0))
    val entity = new GenericEntity[java.util.List[TemplateRevision]](templateRevisions) {}
    Response.ok(entity).build
  }

  @GET
  @Path("/{templateId:.*Release[^/-]*}/revisions/{revisionId}/content")
  def getRevisionContent(@PathParam("templateId") @IdParam templateId: String, @PathParam("revisionId") revisionId: Int): Response = {
    permissionChecker.checkView(templateId)
    val template = templateStorageService.load(revisionId, getResolutionContext(templateId))
    Response.ok(getTemplateAsDsl(template), MediaType.TEXT_PLAIN).build()
  }

  @POST
  @Path("/{templateId:.*Release[^/-]*}/versions")
  def createVersion(@PathParam("templateId") @IdParam templateId: String, versionView: VersionView): Response = {
    checkArgument(releaseService.isTemplate(templateId), "Snapshots can only be created for templates")
    permissionChecker.checkEdit(templateId)
    val user = getAuthenticatedUserName
    val result = for {
      tagName <- Try(templateStorageListener.createNewVersion(templateId, versionView.name, versionView.description, user))
      r: Try[Int] = releaseActorService.executeCommand(templateId, NewTemplateVersion(tagName, versionView.description, user))
    } yield r
    Response.ok(result.fold({
      case e@ScmException(msg, _, statusCode) =>
        logger.warn("Unable to create a new version of a template.", e)
        new CreateVersionResponse(false, msg, statusCode)
      case e: Throwable =>
        logger.warn("Unable to create a new version of a template.", e)
        new CreateVersionResponse(false, e.getMessage, HttpStatus.SC_INTERNAL_SERVER_ERROR)
    }, id => new CreateVersionResponse(true, id.toString))).build()
  }

  @PUT
  @Path("/{templateId:.*Release[^/-]*}/revisions/{revisionId}/restore")
  def restoreRevision(@PathParam("templateId") @IdParam templateId: String, @PathParam("revisionId") revisionId: Int): Response = {
    permissionChecker.check(EDIT_TEMPLATE, templateId)
    permissionChecker.check(CREATE_RELEASE_FROM_TEMPLATE, templateId)
    releaseActorService.executeCommand(templateId, RestoreRevision(revisionId))
    Response.ok().build()
  }

  @DELETE
  @Path("/{templateId:.*Release[^/-]*}/revisions/{revisionId}")
  def deleteRevision(@PathParam("templateId") @IdParam templateId: String, @PathParam("revisionId") revisionId: Int): Response = {
    permissionChecker.checkEdit(templateId)
    templateStorageService.deleteRevision(templateId, revisionId)
    Response.ok().build()
  }

  @DELETE
  @Path("/{templateId:.*Release[^/-]*}/revisions")
  def deleteRevisions(@PathParam("templateId") @IdParam templateId: String, revisionIds: JList[Int]): Response = {
    permissionChecker.checkEdit(templateId)
    templateStorageService.deleteRevisions(templateId, revisionIds.asScala.toSeq)
    Response.ok().build()
  }

  @GET
  @Path("/{templateId:.*Release[^/-]*}/compare")
  def compare(@PathParam("templateId") @IdParam templateId: String,
              @DefaultValue("0") @QueryParam("revisionId") revisionId: Int,
              @DefaultValue("0") @QueryParam("toRevisionId") toRevisionId: Int): TemplateContentAndPrevious = {
    permissionChecker.checkView(templateId)
    val revision = if (revisionId == 0) {
      releaseService.findById(templateId)
    } else {
      templateStorageService.load(revisionId, getResolutionContext(templateId))
    }

    val toRevision = if (toRevisionId == 0) {
      releaseService.findById(templateId)
    } else {
      templateStorageService.load(toRevisionId, getResolutionContext(templateId))
    }

    TemplateContentAndPrevious(current = getTemplateAsDsl(revision), previous = getTemplateAsDsl(toRevision))
  }

  private def getResolutionContext(templateId: String): ResolutionContext = {
    if (Ids.isInFolder(templateId)) {
      ResolutionContext(Ids.getParentId(templateId))
    } else {
      ResolutionContext.GLOBAL
    }
  }

  private def getTemplateAsDsl(template: Release): String = {
    dslExporter.exportTemplate(template)
  }
}
