package com.xebialabs.xlrelease.versioning.templates.repository

import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.repository.sql.persistence.PersistenceSupport
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import com.xebialabs.xlrelease.versioning.templates.repository.persistence.TemplateVersioningSchema.{TEMPLATE_REVISIONS_DATA => TRD}
import com.xebialabs.xlrelease.versioning.templates.repository.persistence.data.TemplateRevisionData
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.jdbc.core.namedparam.{MapSqlParameterSource, NamedParameterJdbcTemplate}
import org.springframework.jdbc.core.support.SqlLobValue
import org.springframework.stereotype.Repository
import org.springframework.util.StreamUtils.copyToByteArray

import java.sql.{ResultSet, Types}
import scala.util.{Failure, Success, Try, Using}

trait TemplateRevisionDataRepository {
  def create(templateRevisionData: TemplateRevisionData): Unit

  def findById(revisionId: Int): Option[TemplateRevisionData]
}

@Repository
@IsTransactional
class SqlTemplateRevisionDataRepository @Autowired()(@Qualifier("xlrRepositoryJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                                                     @Qualifier("xlrRepositorySqlDialect") val dialect: Dialect)
  extends TemplateRevisionDataRepository with PersistenceSupport {

  private val namedJdbcTemplate = new NamedParameterJdbcTemplate(jdbcTemplate)

  override def create(templateRevisionData: TemplateRevisionData): Unit = {
    val stmt =
      s"""INSERT INTO ${TRD.TABLE}
         | (${TRD.REVISION_ID},
         | ${TRD.XLR_VERSION},
         | ${TRD.CONTENT})
         | VALUES (:${TRD.REVISION_ID}, :${TRD.XLR_VERSION}, :${TRD.CONTENT})""".stripMargin
    val params = new MapSqlParameterSource()
    params.addValue(TRD.REVISION_ID, templateRevisionData.revisionId)
    params.addValue(TRD.XLR_VERSION, templateRevisionData.dataModelVersion)
    params.addValue(TRD.CONTENT, new SqlLobValue(templateRevisionData.content), Types.BLOB)
    namedJdbcTemplate.update(stmt, params)
  }

  override def findById(revisionId: Int): Option[TemplateRevisionData] = {
    val query =
      s"""SELECT TRD.${TRD.REVISION_ID}, TRD.${TRD.XLR_VERSION}, TRD.${TRD.CONTENT}
         | FROM ${TRD.TABLE} TRD
         | WHERE TRD.${TRD.REVISION_ID} = :${TRD.REVISION_ID}""".stripMargin
    sqlQuery(query, params(TRD.REVISION_ID -> revisionId), (rs: ResultSet) => {
      Using.resource(rs.getBinaryStream(TRD.CONTENT)) { is =>
        Try(copyToByteArray(is)) match {
          case Success(bytes) => TemplateRevisionData(rs.getInt(TRD.REVISION_ID), rs.getString(TRD.XLR_VERSION), bytes)
          case Failure(ex) => throw new IllegalStateException("Failed to load revision content", ex)
        }
      }
    }).headOption
  }

}

