package com.xebialabs.xlrelease.versioning.templates.repository

import com.xebialabs.xlrelease.db.sql.LimitOffset
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.repository.Ids.getName
import com.xebialabs.xlrelease.repository.Page
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.RELEASES
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.{params, _}
import com.xebialabs.xlrelease.repository.sql.persistence.{CiUid, PersistenceSupport}
import com.xebialabs.xlrelease.versioning.templates.repository.persistence.TemplateVersioningSchema.{TEMPLATE_REVISIONS => TR}
import com.xebialabs.xlrelease.versioning.templates.repository.persistence.data.{TemplateRevision, TemplateRevisionType}
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.stereotype.Repository

import java.sql.ResultSet
import java.util.{List => JList}
import scala.jdk.CollectionConverters._
import scala.util.Try

trait TemplateRevisionRepository {

  def find(templateId: String, page: Page): JList[TemplateRevision]

  def create(templateRevision: TemplateRevision): TemplateRevision

  def latestRevision(templateId: String): Option[TemplateRevision]

  def latestVersion(templateId: String): Option[TemplateRevision]

  def deleteRevisionsForVersion(templateVersion: TemplateRevision): Unit

  def deleteRevisions(templateCiUid: Integer, templateRevisionIds: Seq[Int]): Unit
}

@Repository
@IsTransactional
class SqlTemplateRevisionRepository @Autowired()(@Qualifier("xlrRepositoryJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                                                 @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect)
  extends TemplateRevisionRepository
    with LimitOffset with PersistenceSupport {

  override def find(templateId: String, page: Page): JList[TemplateRevision] = {
    val query =
      s"""SELECT TR.${TR.ID}, TR.${TR.CI_UID}, TR.${TR.REVISION_TYPE}, TR.${TR.VERSION}, TR.${TR.DESCRIPTION}, TR.${TR.USERNAME}, TR.${TR.REVISION_TIME}
         | FROM ${TR.TABLE} TR
         | JOIN ${RELEASES.TABLE} R ON TR.${TR.CI_UID} = R.${RELEASES.CI_UID}
         |WHERE R.${RELEASES.RELEASE_ID} = ?
         |ORDER BY TR.${TR.ID} DESC""".stripMargin
    val paginatedQuery = addLimitAndOffset(query, page.resultsPerPage, page.offset)
    jdbcTemplate.query(paginatedQuery,
      (rs: ResultSet, _: Int) => {
        val templateRevision = new TemplateRevision
        templateRevision.id = rs.getInt(TR.ID)
        templateRevision.ciUid = rs.getInt(TR.CI_UID)
        templateRevision.revisionType = rs.getInt(TR.REVISION_TYPE)
        templateRevision.version = rs.getString(TR.VERSION)
        templateRevision.description = rs.getString(TR.DESCRIPTION)
        templateRevision.username = rs.getString(TR.USERNAME)
        templateRevision.revisionTime = rs.getTimestamp(TR.REVISION_TIME)
        templateRevision
      },
      getName(templateId)
    )
  }

  override def create(templateRevision: TemplateRevision): TemplateRevision = {
    val stmt =
      s"""INSERT INTO ${TR.TABLE} (${TR.CI_UID}, ${TR.REVISION_TYPE}, ${TR.VERSION}, ${TR.DESCRIPTION}, ${TR.USERNAME}, ${TR.REVISION_TIME})
         | VALUES (:${TR.CI_UID}, :${TR.REVISION_TYPE}, :${TR.VERSION}, :${TR.DESCRIPTION}, :${TR.USERNAME}, :${TR.REVISION_TIME})
       """.stripMargin
    sqlInsert(pkName(TR.ID),
      stmt,
      params(TR.CI_UID -> templateRevision.ciUid,
        TR.REVISION_TYPE -> templateRevision.revisionType,
        TR.VERSION -> templateRevision.version.truncateBytes(255),
        TR.DESCRIPTION -> templateRevision.description.truncateBytes(255),
        TR.USERNAME -> templateRevision.username,
        TR.REVISION_TIME -> templateRevision.revisionTime
      ),
      (id: CiUid) => {
        templateRevision.id = id
        templateRevision
      })
  }

  // applicable only when track changes are enabled
  override def latestRevision(templateId: String): Option[TemplateRevision] = latest(templateId)

  override def latestVersion(templateId: String): Option[TemplateRevision] = latest(templateId, version = true)

  private def latest(templateId: String, version: Boolean = false): Option[TemplateRevision] = {
    val query =
      s"""SELECT TR.${TR.ID}, TR.${TR.CI_UID}, TR.${TR.REVISION_TYPE}, TR.${TR.VERSION}, TR.${TR.DESCRIPTION}, TR.${TR.USERNAME}, TR.${TR.REVISION_TIME}
         | FROM ${TR.TABLE} TR
         | WHERE TR.ID = (SELECT MAX(OTR.${TR.ID})
         |                FROM ${TR.TABLE} OTR
         |                JOIN ${RELEASES.TABLE} R ON OTR.${TR.CI_UID} = R.${RELEASES.CI_UID}
         |                WHERE R.${RELEASES.RELEASE_ID} = ?
         |                ${
        if (version) {
          s"AND OTR.${TR.REVISION_TYPE} = 0"
        } else {
          ""
        }
      }
         |                )""".stripMargin
    Try {
      jdbcTemplate.queryForObject(query,
        (rs: ResultSet, _: Int) => {
          val templateRevision = new TemplateRevision
          templateRevision.id = rs.getInt(TR.ID)
          templateRevision.ciUid = rs.getInt(TR.CI_UID)
          templateRevision.revisionType = rs.getInt(TR.REVISION_TYPE)
          templateRevision.version = rs.getString(TR.VERSION)
          templateRevision.description = rs.getString(TR.DESCRIPTION)
          templateRevision.username = rs.getString(TR.USERNAME)
          templateRevision.revisionTime = rs.getTimestamp(TR.REVISION_TIME)
          templateRevision
        },
        getName(templateId)
      )
    }.toOption
  }

  private val STMT_DELETE_REVISIONS_BETWEEN_GIVEN_REVISION_AND_PREVIOUS_VERSION =
    s"""
       |DELETE FROM ${TR.TABLE}
       |WHERE ${TR.ID} IN (
       |  SELECT ${TR.ID} FROM (
       |    SELECT DTR.${TR.ID}
       |    FROM ${TR.TABLE} DTR
       |    WHERE DTR.${TR.REVISION_TYPE} = ${TemplateRevisionType.REVISION}
       |          AND DTR.${TR.CI_UID} = :CI_UID
       |          AND DTR.${TR.ID} < :REV_ID
       |          AND DTR.${TR.ID} > (
       |      SELECT OID FROM (
       |        SELECT COALESCE(MAX(OTR.${TR.ID}), 0) OID
       |        FROM ${TR.TABLE} OTR
       |        WHERE OTR.${TR.REVISION_TYPE} = ${TemplateRevisionType.VERSION}
       |              AND OTR.${TR.CI_UID} = :CI_UID
       |              AND OTR.${TR.ID} < :REV_ID
       |      ) O_ID
       |    )
       |  ) ID_LIST
       |)""".stripMargin

  override def deleteRevisionsForVersion(templateVersion: TemplateRevision): Unit = {
    assert(templateVersion.revisionType == TemplateRevisionType.VERSION)
    sqlExec(STMT_DELETE_REVISIONS_BETWEEN_GIVEN_REVISION_AND_PREVIOUS_VERSION,
      params("CI_UID" -> templateVersion.ciUid, "REV_ID" -> templateVersion.id),
      _.execute())
  }

  private val STMT_DELETE_REVISIONS =
    s"""
       |DELETE FROM ${TR.TABLE}
       |WHERE ${TR.CI_UID} = :RELEASE_UID
       |AND ${TR.ID} IN (:REV_IDS)
     """.stripMargin

  override def deleteRevisions(templateCiUid: Integer, templateRevisionIds: Seq[Int]): Unit = {
    sqlExec(STMT_DELETE_REVISIONS, params("RELEASE_UID" -> templateCiUid, "REV_IDS" -> templateRevisionIds.asJava), _.execute())
  }
}
