package com.xebialabs.xlrelease.versioning.scm

import com.xebialabs.deployit.checks.Checks.ConditionFailedException
import com.xebialabs.deployit.core.rest.resteasy.{WorkDirTemplate, WorkdirHolder}
import com.xebialabs.xlrelease.domain.utils.ScmUtils.normalize
import com.xebialabs.xlrelease.domain.{Attachment, Release}
import com.xebialabs.xlrelease.export.ReleasefileExporter
import com.xebialabs.xlrelease.repository.{FolderRepository, Ids}
import com.xebialabs.xlrelease.scm.connector._
import com.xebialabs.xlrelease.versioning.scm.ScmTemplateToBlobsConverter.FORBIDDEN_CHARS
import org.apache.commons.io.IOUtils
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import scala.jdk.CollectionConverters._

@Service
class ScmTemplateToBlobsConverter @Autowired()(releasefileExporter: ReleasefileExporter,
                                               folderRepository: FolderRepository) {
  final val EXPORT_FOLDER = "XLRelease"
  private val ATTACHMENT_EXPORT_FOLDER = s"${EXPORT_FOLDER}/.${Attachment.EXPORT_DIRECTORY}"

  def convertTemplate(template: Release): (String, ScmBlobs) = {
    val folderId = Ids.getParentId(template.getId)
    val encodedPathWithoutRoot = folderRepository.getPath(folderId).drop(1).map(encode).mkString(Ids.SEPARATOR)
    val folderPath = Ids.SEPARATOR + encodedPathWithoutRoot // path of dsl
    val templatePath = normalizeFilePath(folderPath, encode(template.getTitle) + ".groovy")
    val dsl = BinaryFile(
      templatePath,
      () => releasefileExporter.exportAsString(template, asTemplate = true).getBytes()
    )

    val attachments = template.getAttachments.asScala.map(getAttachmentContent).toSeq

    templatePath -> ScmBlobs(attachments.prepended(dsl))
  }

  private def encode(v: String): String = {
    val filteredString = v.toSeq.filter(!FORBIDDEN_CHARS.contains(_))
    if (filteredString.length != v.length) {
      throw new ConditionFailedException(s"'$v' must not contain any of the special characters: '$FORBIDDEN_CHARS'. Please rename it or disable SCM.")
    }
    v
  }

  private def getAttachmentContent(attachment: Attachment): BinaryFile = {
    BinaryFile(
      ATTACHMENT_EXPORT_FOLDER + "/" + attachment.getExportFilename.replaceFirst(Attachment.EXPORT_DIRECTORY + "/", ""),
      () => {
        WorkdirHolder.initWorkdir(null)
        WorkDirTemplate.cleanOnFinally(_ => IOUtils.toByteArray(attachment.getFile.getInputStream))
      })
  }

  private def normalizeFilePath(folderName: String, fileName: String): String = {
    normalize(s"${EXPORT_FOLDER}/${normalize(folderName)}/${normalize(fileName)}".replaceAll("/+", "/"))
  }
}

object ScmTemplateToBlobsConverter {
  val FORBIDDEN_CHARS = "/:*?\"<>|"
}
