package com.xebialabs.xlrelease.versioning.scm.upgrade

import com.xebialabs.deployit.server.api.upgrade.{Upgrade, Version}
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.exception.LogFriendlyNotFoundException
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import com.xebialabs.xlrelease.repository.sql.persistence.configuration.ReleaseConfigurationReferencePersistence
import com.xebialabs.xlrelease.repository.sql.persistence.data.ReleaseRow
import com.xebialabs.xlrelease.repository.sql.persistence.{PersistenceSupport, ReleasePersistence}
import com.xebialabs.xlrelease.service.ReleaseSearchService
import com.xebialabs.xlrelease.upgrade.Components.XL_RELEASE_COMPONENT
import com.xebialabs.xlrelease.upgrade.UpgradeSupport._
import com.xebialabs.xlrelease.upgrade.{JsonUpgrade, UpgradeResult}
import grizzled.slf4j.Logging
import org.codehaus.jettison.json.JSONObject
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.stereotype.Component
import org.springframework.transaction.support.TransactionTemplate

import java.util.Collections
import scala.jdk.CollectionConverters._

@Component
class XLRelease820ScmConnectedTemplateRemovalUpgrade @Autowired()(@Qualifier("xlrRepositoryJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                                                                  @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect,
                                                                  @Qualifier("xlrRepositoryTransactionTemplate") val transactionTemplate: TransactionTemplate,
                                                                  releaseSearchService: ReleaseSearchService,
                                                                  releasePersistence: ReleasePersistence,
                                                                  jsonUpgrade: XLRelease820ScmConnectedTemplateRemovalJsonUpgrade)
  extends Upgrade with PersistenceSupport with Logging with BatchSupport with TransactionSupport {

  override def upgradeVersion(): Version = Version.valueOf(XL_RELEASE_COMPONENT, "8.2.0#2")

  override def doUpgrade(): Boolean = {
    logger.info("Starting SCM connected template removal upgrade")
    val items = releaseSearchService.findAllTemplateIds.asScala
    doInBatch(items) { batch =>
      doInTransaction {
        batch.items.foreach(releaseId => {
          val releaseRow: Option[ReleaseRow] = releasePersistence.findByReleaseId(releaseId)
          releaseRow.foreach((row: ReleaseRow) => {
            logger.trace(s"Processing row with CI_UID: [${row.ciUid}] that belongs to template [$releaseId]")
            val releaseJson = new JSONObject(row.json)
            jsonUpgrade.performUpgrade(releaseJson)
            sqlExecWithContent(ReleasePersistence.STMT_UPDATE_RELEASE_CONTENT, params("ciUid" -> row.ciUid), "content" -> releaseJson.toString, {
              case 0 => throw new LogFriendlyNotFoundException(s"Template [$releaseId] could not be updated.")
              case _ =>
            })
          })
        })
      }
    }
    logger.info("Finished SCM connected template removal upgrade")
    true
  }

}

@Component
class XLRelease820ScmConnectedTemplateRemovalJsonUpgrade extends JsonUpgrade {
  override def upgradeVersion(): Version = Version.valueOf(XL_RELEASE_COMPONENT, "8.2.0#3")

  override def performUpgrade(release: JSONObject): UpgradeResult = {
    val releaseType = release.getString("type")
    val upgraded = if (releaseType == "xlrelease.ScmConnectedTemplate") {
      release.put("type", "xlrelease.Release")
      release.remove("scmConnectorConfig")
      true
    } else {
      false
    }
    new UpgradeResult(upgraded, Collections.emptyList())
  }
}

class XLRelease820ScmConnectorConfigRemovalUpgrade @Autowired()(@Qualifier("xlrRepositoryTransactionTemplate") val transactionTemplate: TransactionTemplate,
                                                                configurationPersistence: ReleaseConfigurationReferencePersistence)
  extends Upgrade with Logging with BatchSupport with TransactionSupport {

  override def upgradeVersion(): Version = Version.valueOf(XL_RELEASE_COMPONENT, "8.2.0#4")

  override def doUpgrade(): Boolean = {
    logger.info("Starting SCM connector config removal upgrade")

    val items = configurationPersistence.findUidsByTypes(Seq("xlrelease.GitHubScmConnectorConfig"))
    doInBatch(items) { batch =>
      doInTransaction {
        batch.items.foreach(configurationUid => {
          logger.trace(s"Removing SCM connector and references with CI_UID: $configurationUid")
          configurationPersistence.deleteRefsByConfigurationUid(configurationUid)
          configurationPersistence.deleteByUid(configurationUid)
        })
      }
    }

    logger.info("Finished SCM connector config removal upgrade")
    true
  }
}



