package com.xebialabs.xlrelease.triggers.activity

import com.xebialabs.xlrelease.activity.{ActivityLogger, LoggingParams, ReleaseActivityLoggingEventHandler}
import com.xebialabs.xlrelease.domain.TriggerActivity._
import com.xebialabs.xlrelease.domain.events._
import com.xebialabs.xlrelease.events.{EventListener, Subscribe, XLReleaseEventBus}
import com.xebialabs.xlrelease.repository.ActivityLogRepository
import com.xebialabs.xlrelease.triggers.events._
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

@Component
@EventListener
class TriggerLoggingEventHandler @Autowired()(val eventBus: XLReleaseEventBus, val activityLogRepository: ActivityLogRepository)
  extends ActivityLogger[TriggerEvent] {

  @Subscribe
  def onEvent(event: TriggerEvent): Unit = log(event)

  // scalastyle:off cyclomatic.complexity
  override def logEntries: PartialFunction[XLReleaseEvent, LoggingParams] = {
    case ev@TriggerCreatedEvent(trigger) =>
      LoggingParams(Some(trigger.getId), None, TRIGGER_CREATED.create(ev, trigger, trigger.getTitle))

    case ev@TriggerUpdatedEvent(original, updated) =>
      val logs = new TriggerFieldsComparator(ev.timestamp, ev.username, original, updated).getLogs
      LoggingParams(Some(updated.getId), Option(ev.username), logs)

    case ev@TriggerDeletedEvent(trigger) =>
      LoggingParams(Some(trigger.getId), None, TRIGGER_DELETED.create(ev, trigger, trigger.getTitle))

    case ev@TriggerEnabledEvent(trigger) =>
      LoggingParams(Some(trigger.getId), None, TRIGGER_ENABLED.create(ev, trigger, trigger.getTitle))

    case ev@TriggerDisabledEvent(trigger) =>
      LoggingParams(Some(trigger.getId), None, TRIGGER_DISABLED.create(ev, trigger, trigger.getTitle))

    case ev@TriggerExecutedEvent(trigger, actionResult, dataId) =>
      val entry = TRIGGER_EXECUTED.create(ev, trigger, actionResult, trigger.getTitle)
      entry.setDataId(dataId)
      LoggingParams(Some(trigger.getId), None, entry)

    case ev@TriggerSkippedEvent(trigger, message, dataId) =>
      val entry = TRIGGER_SKIPPED.create(ev, trigger, trigger.getTitle, message)
      entry.setDataId(dataId)
      LoggingParams(Some(trigger.getId), None, entry)

    case ev@TriggerFailedEvent(trigger, error, dataId) =>
      val entry = TRIGGER_FAILED.create(ev, trigger, trigger.getTitle, stripException(error))
      entry.setDataId(dataId)
      LoggingParams(Some(trigger.getId), None, entry)

    case ev@TriggerFilterFailedEvent(trigger, error, dataId) =>
      val entry = TRIGGER_FILTER_FAILED.create(ev, trigger, trigger.getTitle, stripException(error))
      entry.setDataId(dataId)
      LoggingParams(Some(trigger.getId), None, entry)

    case ev@TriggerCreatedFromAsCodeEvent(trigger, scmTraceabilityData) =>
      LoggingParams(Some(trigger.getId), None, TRIGGER_CREATED_FROM_AS_CODE.
        create(ev, trigger, ReleaseActivityLoggingEventHandler.getScmTraceabilityDataInfo(scmTraceabilityData)))

    case ev@TriggerUpdatedFromAsCodeEvent(updated, scmTraceabilityData) =>
      LoggingParams(Some(updated.getId), None, TRIGGER_PROPERTY_UPDATED_FROM_AS_CODE.
        create(ev, updated, ReleaseActivityLoggingEventHandler.getScmTraceabilityDataInfo(scmTraceabilityData)))

  }

  private def stripException(message: String): String =
    Option(message).map(_.replaceFirst(".*Exception:\\s", "")).getOrElse("")
}
