package com.xebialabs.xlrelease.triggers.repository

import com.codahale.metrics.annotation.Timed
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.api.v1.filter.TriggerFilters
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.domain.Trigger
import com.xebialabs.xlrelease.repository.sql.InterceptedRepository
import com.xebialabs.xlrelease.repository.sql.persistence.CiId._
import com.xebialabs.xlrelease.repository.sql.persistence.CiUid
import com.xebialabs.xlrelease.repository.{Ids, RemoveTriggeredReleaseCondition, TriggerRepository}
import com.xebialabs.xlrelease.service.CiIdService
import com.xebialabs.xlrelease.triggers.repository.TriggerPersistenceHelper._
import com.xebialabs.xlrelease.triggers.repository.persistence.{TriggerPersistence, TriggeredReleaseRow}
import grizzled.slf4j.Logging
import org.springframework.data.domain.{Page, Pageable}

import scala.jdk.CollectionConverters._

@IsTransactional
class SqlTriggerRepository(val triggerPersistence: TriggerPersistence,
                           val ciIdService: CiIdService)
  extends TriggerRepository with InterceptedRepository[Trigger] with Logging {


  @Timed
  @IsReadOnly
  override def find[T <: Trigger](triggerId: String): T = {
    val maybeTrigger = if (Ids.isInRelease(triggerId)) {
      triggerPersistence.findById[T](Ids.getFolderlessId(triggerId.normalized))
    } else {
      triggerPersistence.findById[T](Ids.getName(triggerId))
    }
    maybeTrigger.getOrElse {
      throw new NotFoundException("Repository entity [%s] not found", triggerId)
    }
  }

  @Timed
  override def create[T <: Trigger](trigger: T): T = {
    if (!Ids.isTriggerId(trigger.getId)) {
      throw new IllegalArgumentException(s"Trigger Id [${trigger.getId}] is not valid")
    }
    fixPropertyIds(trigger, ciIdService)
    interceptCreate(trigger)
    triggerPersistence.insert(trigger)
    afterCreate(trigger)
    trigger
  }

  @Timed
  override def update[T <: Trigger](trigger: T): T = {
    fixPropertyIds(trigger, ciIdService)
    interceptUpdate(trigger)
    triggerPersistence.update(trigger)
    afterUpdate(trigger)
    trigger
  }

  @Timed
  override def delete(triggerId: String): Unit = {
    interceptDelete(triggerId)
    triggerPersistence.delete(Ids.getFolderlessId(triggerId.normalized))
    afterDelete(triggerId)
  }

  @Timed
  @IsReadOnly
  override def findByFolderId(folderId: String, nestedFolders: Boolean, pageable: Pageable): Page[Trigger] = {
    val filter = new TriggerFilters()
    filter.setFolderId(folderId)
    filter.setNestedFolders(nestedFolders)
    triggerPersistence.findBy(filter, pageable)
  }

  @Timed
  @IsReadOnly
  override def numberOfTemplateTriggers(templateId: String): Int = {
    triggerPersistence.numberOfTemplateTriggers(templateId)
  }

  @Timed
  @IsReadOnly
  override def findByTemplateId(templateId: String, pageable: Pageable): Page[Trigger] = {
    val filter = new TriggerFilters()
    filter.setTemplateId(templateId)
    triggerPersistence.findBy(filter, pageable)
  }

  @Timed
  @IsReadOnly
  override def findByType(triggerType: List[Type], pageable: Pageable): Page[Trigger] = {
    val filter = new TriggerFilters()
    filter.setTriggerType(triggerType.map(_.toString).asJava)
    triggerPersistence.findBy(filter, pageable)
  }

  @Timed
  @IsReadOnly
  override def getRunningTriggeredReleasesCount(templateCiUid: CiUid): Int = {
    val total = triggerPersistence.getRunningTriggeredReleasesCount(templateCiUid)
    logger.debug(s"getRunningTriggeredReleasesCount: templateCiUid=$templateCiUid, total=$total")
    total
  }

  @Timed
  override def insertTriggeredRelease(templateUid: CiUid, triggerUid: CiUid, triggeredReleaseUid: CiUid): Unit = {
    logger.debug(s"insertTriggeredRelease: templateUid=$templateUid, triggerUid=$triggerUid, triggeredReleaseUid=$triggeredReleaseUid")
    val row = TriggeredReleaseRow(templateUid, triggerUid, triggeredReleaseUid)
    triggerPersistence.insertTriggeredReleaseRow(row)
  }

  @Timed
  override def removeTriggeredRelease(condition: RemoveTriggeredReleaseCondition): Unit = {
    logger.debug(s"removeTriggeredRelease: $condition")
    triggerPersistence.removeTriggeredReleasesRow(condition)
  }
}
