package com.xebialabs.xlrelease.triggers

import com.xebialabs.xlrelease.domain.events.{ReleaseDeletedEvent, ReleaseExecutedEvent}
import com.xebialabs.xlrelease.events.{Subscribe, XLReleaseEventBus}
import com.xebialabs.xlrelease.repository.{RemoveByReleaseUid, RemoveByTemplateUid, TriggerRepository}
import grizzled.slf4j.Logging
import org.springframework.context.annotation.Lazy
import org.springframework.stereotype.Component

import javax.annotation.{PostConstruct, PreDestroy}
import scala.util.Try

@Component
@Lazy(false)
class ReleaseTriggerSubscriber(triggerRepository: TriggerRepository, eventBus: XLReleaseEventBus) extends Logging {

  @PostConstruct
  def postConstruct(): Unit = {
    eventBus.register(this)
  }

  @PreDestroy
  def preDestroy(): Unit = {
    eventBus.deregister(this)
  }

  @Subscribe
  def templateDeleted(releaseDeletedEvent: ReleaseDeletedEvent): Unit = {
    val releaseDeleted = releaseDeletedEvent.release
    if (releaseDeleted.isTemplate) {
      Try(triggerRepository.removeTriggeredRelease(RemoveByTemplateUid(releaseDeleted.getCiUid)))
    }
  }

  @Subscribe
  def releaseExecuted(releaseExecutedEvent: ReleaseExecutedEvent): Unit = {
    val executedRelease = releaseExecutedEvent.release
    if (executedRelease.isCreatedFromTrigger) {
      Try(triggerRepository.removeTriggeredRelease(RemoveByReleaseUid(executedRelease.getCiUid)))
    }
  }
}
