package com.xebialabs.xlrelease.api.internal;

import java.util.List;
import jakarta.ws.rs.*;
import jakarta.ws.rs.core.MediaType;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Controller;

import com.xebialabs.xlrelease.triggers.activity.TriggerActivityLogsService;
import com.xebialabs.xlrelease.triggers.security.TriggerPermissionChecker;
import com.xebialabs.xlrelease.views.ActivityLogEntryView;
import com.xebialabs.xlrelease.views.LogsFilters;
import com.xebialabs.xlrelease.views.converters.UserViewConverter;

import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.xlrelease.api.ApiService.DEFAULT_RESULTS_PER_PAGE;
import static com.xebialabs.xlrelease.api.ApiService.PAGE;
import static com.xebialabs.xlrelease.api.ApiService.RESULTS_PER_PAGE;

@Path("/activity")
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
@Controller
public class TriggerActivityLogsResource extends BaseActivityLogsResource {

    private final TriggerPermissionChecker triggerPermissionChecker;
    private final TriggerActivityLogsService triggerActivityLogsService;

    @Autowired
    public TriggerActivityLogsResource(final TriggerActivityLogsService triggerActivityLogsService,
                                       final UserViewConverter userViewConverter,
                                       final TriggerPermissionChecker triggerPermissionChecker) {
        super(triggerActivityLogsService, userViewConverter);
        this.triggerPermissionChecker = triggerPermissionChecker;
        this.triggerActivityLogsService = triggerActivityLogsService;
    }

    /**
     * Searches in the activity logs of a trigger.
     *
     * @param triggerId the identifier of the trigger
     * @param filters   the search criteria to use
     * @return the activity logs
     */
    @POST
    @Path("/{triggerId:.*Trigger[^/-]*}/logs/search")
    public List<ActivityLogEntryView> searchTriggerLogs(
            @PathParam("triggerId") String triggerId,
            @DefaultValue("0") @QueryParam(PAGE) int page,
            @DefaultValue("100") @QueryParam(RESULTS_PER_PAGE) int resultsPerPage,
            LogsFilters filters
    ) {
        checkArgument(resultsPerPage <= DEFAULT_RESULTS_PER_PAGE, "Number of results per page cannot be more than %d", DEFAULT_RESULTS_PER_PAGE);
        triggerPermissionChecker.checkViewTrigger(triggerId);

        // sort feature of Pageable is not used, because there is sort order in LogsFilters
        PageRequest pageable = PageRequest.of(page, resultsPerPage, Sort.unsorted());
        return searchLogs(triggerId, filters, pageable);
    }

    @GET
    @Path("/data/{dataId}")
    public String getTriggerActivityLogEntryData(@PathParam("dataId") String dataId) {
        return triggerActivityLogsService.getTriggerActivityLogData(dataId);
    }
}
