package com.xebialabs.xlrelease.triggers.management.repository

import com.xebialabs.deployit.security.permission.Permission
import com.xebialabs.xlrelease.api.v1.filter.TriggerFilters
import com.xebialabs.xlrelease.db.sql.SqlBuilder._
import com.xebialabs.xlrelease.domain.status.TriggerExecutionStatus
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.repository.Ids.{SEPARATOR, getName}
import com.xebialabs.xlrelease.repository.query._
import com.xebialabs.xlrelease.repository.sql.persistence.SecurableSqlBuilder
import com.xebialabs.xlrelease.security.XLReleasePermissions._
import com.xebialabs.xlrelease.security.sql.db.SecuritySchema.{ROLES, ROLE_PERMISSIONS, ROLE_PRINCIPALS, ROLE_ROLES}
import com.xebialabs.xlrelease.utils.FolderId
import com.xebialabs.xlrelease.utils.QueryParamUtils.getRoleIdParams
import com.xebialabs.xlrelease.view.ReleaseTriggerOverview
import org.springframework.jdbc.core.RowMapper
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate
import org.springframework.util.StringUtils

import java.sql.ResultSet
import scala.jdk.CollectionConverters._

class SqlTriggerFiltersQueryBuilder(val dialect: Dialect,
                                    val namedTemplate: NamedParameterJdbcTemplate)
  extends FiltersQueryBuilder[TriggerFilters, ReleaseTriggerOverview]
    with FilterQueryBuilderSupport[TriggerFilters, ReleaseTriggerOverview] {

  import com.xebialabs.xlrelease.repository.sql.persistence.Schema._

  private var hasEditableFlag = false

  private def rowMapper: RowMapper[ReleaseTriggerOverview] = (rs: ResultSet, _: Int) => {
    val rawTriggerId = rs.getString(TRIGGERS.ID)
    val rawReleaseId = rs.getString(RELEASES.RELEASE_ID)
    val rawFolderId = rs.getString(FOLDERS.FOLDER_ID)
    val rawFolderPath = rs.getString(FOLDERS.FOLDER_PATH)
    val folderId = s"$rawFolderPath$SEPARATOR$rawFolderId".dropWhile(_ == '/')
    val templateId = s"$rawFolderPath$SEPARATOR$rawFolderId$SEPARATOR$rawReleaseId".dropWhile(_ == '/')
    // rawTriggerId contains release id as a prefix
    val triggerId = s"$rawFolderPath$SEPARATOR$rawFolderId$SEPARATOR$rawTriggerId".dropWhile(_ == '/')
    new ReleaseTriggerOverview(
      FolderId(triggerId).absolute,
      rs.getString(TRIGGERS.TRIGGER_TITLE),
      rs.getString(TRIGGERS.DESCRIPTION),
      rs.getString(TRIGGERS.CI_TYPE),
      rs.getBoolean(TRIGGERS.ENABLED),
      rs.getString(RELEASES.RELEASE_TITLE),
      FolderId(templateId).absolute,
      FolderId(if (folderId == null) Ids.ROOT_FOLDER_ID else folderId).absolute,
      rs.getString(FOLDERS.NAME),
      !hasEditableFlag || rs.getString("hasEditOnTemplatePermission") != null,
      rs.getTimestamp(TRIGGERS.LAST_RUN_DATE),
      Option(rs.getString(TRIGGERS.LAST_RUN_STATUS)).map(value => TriggerExecutionStatus.valueOf(value.toUpperCase)).orNull
    )
  }

  private lazy val permissionColumns =
    s"editOnTemplate.${ROLE_PERMISSIONS.ciId} hasEditOnTemplatePermission"

  private lazy val queryTemplate =
    s"""
       |SELECT
       | t.${TRIGGERS.ID}, t.${TRIGGERS.TRIGGER_TITLE},
       | t.${TRIGGERS.DESCRIPTION}, t.${TRIGGERS.CI_TYPE}, t.${TRIGGERS.ENABLED},t.${TRIGGERS.LAST_RUN_DATE},t.${TRIGGERS.LAST_RUN_STATUS},
       | r.${RELEASES.RELEASE_TITLE}, r.${RELEASES.RELEASE_ID},
       | f.${FOLDERS.FOLDER_ID}, f.${FOLDERS.NAME}, f.${FOLDERS.FOLDER_PATH}
       | ${if (hasEditableFlag) "," + permissionColumns else ""}
       |FROM
       | ${TRIGGERS.TABLE} t
       | LEFT JOIN ${FOLDERS.TABLE} f ON f.${FOLDERS.CI_UID} = t.${TRIGGERS.FOLDER_UID}
       | LEFT JOIN ${TEMPLATE_TRIGGERS.TABLE} rt ON t.${TRIGGERS.CI_UID} = rt.${TEMPLATE_TRIGGERS.TRIGGER_UID}
       | LEFT JOIN ${RELEASES.TABLE} r ON r.${RELEASES.CI_UID} = rt.${TEMPLATE_TRIGGERS.RELEASE_UID}
       |$joinClause
       |$whereClause
       |$orderClause
    """.stripMargin.linesIterator.filter(!_.trim.isEmpty).mkString(NL)

  private lazy val totalQueryTemplate =
    s"""
       |SELECT COUNT(1)
       |FROM
       | ${TRIGGERS.TABLE} t
       | LEFT JOIN ${FOLDERS.TABLE} f ON f.${FOLDERS.CI_UID} = t.${TRIGGERS.FOLDER_UID}
       | LEFT JOIN ${TEMPLATE_TRIGGERS.TABLE} rt ON t.${TRIGGERS.CI_UID} = rt.${TEMPLATE_TRIGGERS.TRIGGER_UID}
       | LEFT JOIN ${RELEASES.TABLE} r ON r.${RELEASES.CI_UID} = rt.${TEMPLATE_TRIGGERS.RELEASE_UID}
       |$whereClause
    """.stripMargin.linesIterator.filter(!_.trim.isEmpty).mkString(s"$NL")

  private def selectCiIdsWithPermission(roleIds: Iterable[String], permissionsParameterName: String): String = {
    s"""|SELECT
        | DISTINCT teamPermissions.${ROLE_PERMISSIONS.ciId}
        |FROM
        | ${ROLES.TABLE} teams
        | LEFT JOIN ${ROLE_ROLES.TABLE} teamRoles ON teams.${ROLES.id} = teamRoles.${ROLE_ROLES.roleId}
        | LEFT JOIN ${ROLE_PRINCIPALS.TABLE} teamPrincipals ON teams.${ROLES.id} = teamPrincipals.${ROLE_PRINCIPALS.roleId}
        | JOIN ${ROLE_PERMISSIONS.TABLE} teamPermissions ON teams.${ROLES.id} = teamPermissions.${ROLE_PERMISSIONS.roleId}
        |WHERE
        | teamPermissions.${ROLE_PERMISSIONS.permissionName} IN (:${permissionsParameterName})
        | AND (
        |   teamPrincipals.${ROLE_PRINCIPALS.principalName} IN (:principalNames)
        |   ${SecurableSqlBuilder.createMemberRoleIdInRoleIdsClause(roleIds)}
        | )""".stripMargin.linesIterator.filter(_.trim.nonEmpty).mkString(s"$NL")
  }

  override def from(filter: TriggerFilters): SelfType = {
    if (StringUtils.hasText(filter.getTemplateId)) {
      whereClauses += s"r.${RELEASES.RELEASE_ID} = :templateId"
      queryParams += "templateId" -> Ids.getName(filter.getTemplateId)
    }
    if (StringUtils.hasText(filter.getTriggerTitle)) {
      like(s"t.${TRIGGERS.TRIGGER_TITLE}", "triggerTitle", filter.getTriggerTitle)
    }
    if (StringUtils.hasText(filter.getFolderId)) {
      equal(s"f.${FOLDERS.FOLDER_ID}", "folderId", getName(filter.getFolderId))
    }
    if (StringUtils.hasText(filter.getFolderTitle)) {
      like(s"f.${FOLDERS.NAME}", "folderTitle", filter.getFolderTitle)
    }
    if (StringUtils.hasText(filter.getTemplateTitle)) {
      like(s"r.${RELEASES.RELEASE_TITLE}", "templateTitle", filter.getTemplateTitle)
    }
    if (StringUtils.hasText(filter.getTemplateId)) {
      equal(s"r.${RELEASES.RELEASE_ID}", "templateId", getName(filter.getTemplateId))
    }
    if (!filter.getTriggerType.isEmpty) {
      val triggerTypeQueries = filter.getTriggerType.asScala.zipWithIndex.map {
        case (triggerType, index) => s"t.${TRIGGERS.CI_TYPE} = :triggerType${index}" -> index
      }
      whereClauses += triggerTypeQueries.unzip._1.mkString("(", " OR ", ")")
      for (param <- triggerTypeQueries.unzip._2.map(index => s"triggerType$index" -> filter.getTriggerType.get(index))) {
        queryParams += param
      }
    }
    this
  }

  def withEditableFlag(principals: Iterable[String], roleIds: Iterable[String]): SelfType = {
    hasEditableFlag = true
    joinClauses +=
      s"""LEFT JOIN (${selectCiIdsWithPermission(roleIds, "editOnTemplatePermission")}) editOnTemplate
         |    ON r.${RELEASES.SECURITY_UID} = editOnTemplate.${ROLE_PERMISSIONS.ciId}
       """.stripMargin
    queryParams += "editOnTemplatePermission" -> List(EDIT_TRIGGER.getPermissionName, EDIT_TEMPLATE_TRIGGERS.getPermissionName).asJava
    queryParams += "principalNames" -> principals.toList.asJava
    queryParams = queryParams ++ getRoleIdParams(roleIds)
    this
  }

  override def withPermissions(permissions: Seq[Permission], principals: Iterable[String], roleIds: Iterable[String]): SelfType = {
    whereClauses +=
      s"""r.${RELEASES.SECURITY_UID} IN (
         |  ${selectCiIdsWithPermission(roleIds, "permissionNames")}
         |)
      """.stripMargin
    queryParams += "permissionNames" -> permissions.map(permission => permission.getPermissionName).toList.asJava
    queryParams += "principalNames" -> principals.toList.asJava
    queryParams = queryParams ++ getRoleIdParams(roleIds)
    this
  }

  override def build(): PageableQuery[ReleaseTriggerOverview] = {
    remapSortOrderParams()
    buildSortOrderClause("triggerTitle")
    val resultsQueryString = pageableQuery(queryTemplate)
    val resultsQuery = new SqlListQuery[ReleaseTriggerOverview](namedTemplate, resultsQueryString, queryParams.toMap, rowMapper)
    val countQueryString = totalQueryTemplate
    val totalCountQuery = new SqlQuery[Long](namedTemplate, countQueryString, queryParams.toMap, (rs, _) => rs.getLong(1))
    new SqlPageableQuery[ReleaseTriggerOverview](namedTemplate, this.pageable, resultsQuery, totalCountQuery)
  }

  private def remapSortOrderParams(): Unit = {
    this.withSortParameters(
      "triggerTitle" -> s"t.${TRIGGERS.TRIGGER_TITLE}",
      "triggerDescription" -> s"t.${TRIGGERS.DESCRIPTION}",
      "triggerType" -> s"t.${TRIGGERS.CI_TYPE}",
      "triggerEnabled" -> s"t.${TRIGGERS.ENABLED}",
      "lastRunDate" -> s"t.${TRIGGERS.LAST_RUN_DATE}",
      "templateTitle" -> s"r.${RELEASES.RELEASE_TITLE}",
      "folderTitle" -> s"f.${FOLDERS.NAME}",
      "templateId" -> s"r.${RELEASES.RELEASE_ID}"
    )
  }
}
