package com.xebialabs.xlrelease.triggers.management.repository

import com.xebialabs.deployit.security.permission.Permission
import com.xebialabs.xlrelease.db.sql.SqlBuilder._
import com.xebialabs.xlrelease.repository.Ids.SEPARATOR
import com.xebialabs.xlrelease.repository.query.{Query, SqlListQuery}
import com.xebialabs.xlrelease.repository.sql.persistence.SecurableSqlBuilder
import com.xebialabs.xlrelease.security.sql.db.SecuritySchema.{ROLES, ROLE_PERMISSIONS, ROLE_PRINCIPALS, ROLE_ROLES}
import com.xebialabs.xlrelease.utils.FolderId
import com.xebialabs.xlrelease.utils.QueryParamUtils.getRoleIdParams
import grizzled.slf4j.Logging
import org.springframework.jdbc.core.RowMapper
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate

import java.sql.ResultSet
import scala.collection.mutable
import scala.jdk.CollectionConverters._

class SqlTriggerListQueryBuilder(val dialect: Dialect,
                                 val namedTemplate: NamedParameterJdbcTemplate)
  extends TriggerListQueryBuilder[java.util.List[String]] with Logging {


  import com.xebialabs.xlrelease.repository.sql.persistence.Schema._

  private var queryParams: mutable.Map[String, Any] = mutable.HashMap()
  private val whereClauses: mutable.Buffer[String] = mutable.Buffer()
  private val NL = System.lineSeparator

  private def whereClause: String = {
    if (whereClauses.isEmpty) {
      ""
    }
    else {
      s"WHERE ${whereClauses.mkString(" AND ")}"
    }
  }

  private lazy val listQueryTemplate =
    s"""
       |SELECT
       | t.${TRIGGERS.ID}, f.${FOLDERS.FOLDER_ID}, f.${FOLDERS.FOLDER_PATH}
       |FROM
       | ${TRIGGERS.TABLE} t
       | LEFT JOIN ${FOLDERS.TABLE} f ON f.${FOLDERS.CI_UID} = t.${TRIGGERS.FOLDER_UID}
       | LEFT JOIN ${TEMPLATE_TRIGGERS.TABLE} rt ON t.${TRIGGERS.CI_UID} = rt.${TEMPLATE_TRIGGERS.TRIGGER_UID}
       |$whereClause
    """.stripMargin.linesIterator.filter(_.trim.nonEmpty).mkString(s"$NL")

  private def selectCiIdsWithPermission(roleIds: Iterable[String]): String = {
    s"""|SELECT
        | DISTINCT teamPermissions.${ROLE_PERMISSIONS.ciId}
        |FROM
        | ${ROLES.TABLE} teams
        | LEFT JOIN ${ROLE_ROLES.TABLE} teamRoles ON teams.${ROLES.id} = teamRoles.${ROLE_ROLES.roleId}
        | LEFT JOIN ${ROLE_PRINCIPALS.TABLE} teamPrincipals ON teams.${ROLES.id} = teamPrincipals.${ROLE_PRINCIPALS.roleId}
        | JOIN ${ROLE_PERMISSIONS.TABLE} teamPermissions ON teams.${ROLES.id} = teamPermissions.${ROLE_PERMISSIONS.roleId}
        |WHERE
        | teamPermissions.${ROLE_PERMISSIONS.permissionName} IN (:permissionNames)
        | AND (
        |   teamPrincipals.${ROLE_PRINCIPALS.principalName} IN (:principalNames)
        |   ${SecurableSqlBuilder.createMemberRoleIdInRoleIdsClause(roleIds)}
        | )""".stripMargin.linesIterator.filter(!_.trim.isEmpty).mkString(s"$NL")
  }

  override def withPermissions(permissions: Seq[Permission], principals: Iterable[String], roleIds: Iterable[String]): selfType = {
    whereClauses +=
      s"""f.${FOLDERS.SECURITY_UID} IN (
         |  ${selectCiIdsWithPermission(roleIds)}
         |)
      """.stripMargin
    queryParams += "permissionNames" -> permissions.map(permission => permission.getPermissionName).toList.asJava
    queryParams += "principalNames" -> principals.toList.asJava
    queryParams = queryParams ++ getRoleIdParams(roleIds)
    this
  }

  private def rowMapper: RowMapper[String] = (rs: ResultSet, _: Int) => {
    val rawTriggerId = rs.getString(TRIGGERS.ID)
    val rawFolderId = rs.getString(FOLDERS.FOLDER_ID)
    val rawFolderPath = rs.getString(FOLDERS.FOLDER_PATH)
    val triggerId = s"$rawFolderPath$SEPARATOR$rawFolderId$SEPARATOR$rawTriggerId".dropWhile(_ == '/')
    FolderId(triggerId).absolute
  }

  override def build(): Query[java.util.List[String]] = {
    new SqlListQuery[String](namedTemplate, listQueryTemplate + s"${NL}ORDER BY t.${TRIGGERS.CI_UID} ASC", queryParams.toMap, rowMapper)
  }
}

trait TriggerListQueryBuilder[R] {
  type selfType = TriggerListQueryBuilder[R]

  def withPermissions(permissions: Seq[Permission], principals: Iterable[String], roleIds: Iterable[String]): selfType

  def build(): Query[R]
}

object TriggerListQueryBuilder {
  def apply(dialect: Dialect, namedTemplate: NamedParameterJdbcTemplate): TriggerListQueryBuilder[java.util.List[String]] =
    new SqlTriggerListQueryBuilder(dialect, namedTemplate)
}
