package com.xebialabs.xlrelease.triggers.events.handler

import com.xebialabs.deployit.ServerConfiguration
import com.xebialabs.xlrelease.configuration.FeatureSettings
import com.xebialabs.xlrelease.domain.TriggerActivity._
import com.xebialabs.xlrelease.domain.{ActivityLogEntry, Trigger, TriggerActivity}
import com.xebialabs.xlrelease.events.{AsyncSubscribe, EventListener, XLReleaseEventBus}
import com.xebialabs.xlrelease.notifications.{GenericSystemNotification, NotificationService}
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.service.ConfigurationService
import com.xebialabs.xlrelease.triggers.activity.TriggerActivityLogsService
import com.xebialabs.xlrelease.triggers.events.{TriggerAutoDisabledEvent, TriggerFailureEvent}
import com.xebialabs.xlrelease.triggers.service.TriggerService
import com.xebialabs.xlrelease.utils.Limits.{CONSECUTIVE_TRIGGER_FAILURES, ENABLED, TYPE_LIMITS}
import com.xebialabs.xlrelease.views.LogsFilters
import grizzled.slf4j.Logging
import org.springframework.data.domain.Pageable
import org.springframework.stereotype.Component

import java.util.{List => JList}


@Component
@EventListener
class TriggerFailureEventHandler(triggerService: TriggerService,
                                 triggerActivityLogsService: TriggerActivityLogsService,
                                 configurationService: ConfigurationService,
                                 eventBus: XLReleaseEventBus,
                                 notificationService: NotificationService) extends Logging {

  private lazy val countableTriggerActivityOps = List(TRIGGER_ENABLED, TRIGGER_DISABLED, TRIGGER_EXECUTED, TRIGGER_FAILED, TRIGGER_FILTER_FAILED)
  private lazy val triggerFailureEvents = List(TRIGGER_FAILED, TRIGGER_FILTER_FAILED)

  @AsyncSubscribe
  def onTriggerFailureEvent(event: TriggerFailureEvent): Unit = {
    logger.trace(s"Received TriggerFailureEvent event for trigger [${event.trigger.getId}]")
    checkIfConsecutiveTriggerFailuresLimitReached(event)
  }

  private def checkIfConsecutiveTriggerFailuresLimitReached(event: TriggerFailureEvent): Unit = {
    // TODO: we need to fetch feature limits each time because those can change on any event?
    val limits: FeatureSettings = configurationService.getFeatureSettings(TYPE_LIMITS)
    val limitsConsecutiveTriggerFailures: Int = limits.getProperty(CONSECUTIVE_TRIGGER_FAILURES)
    val limitsEnabled: Boolean = limits.getProperty(ENABLED)

    if (limitsEnabled) {
      val pageable = Pageable.ofSize(limitsConsecutiveTriggerFailures)
      val countableTriggerActivityLogEntries = getCountableTriggerActivityLogEntries(event.trigger.getId, pageable)

      val isConsecutiveTriggerFailuresLimitReachable = countableTriggerActivityLogEntries.size() >= limitsConsecutiveTriggerFailures
      val allTriggerEventsAreFailures = checkIfAllTriggerEventsAreFailures(countableTriggerActivityLogEntries)

      if (isConsecutiveTriggerFailuresLimitReachable && allTriggerEventsAreFailures) {
        logger.warn(s"Trigger [${event.trigger.getId}] reached the maximum configured number of consecutive failures: ${limitsConsecutiveTriggerFailures}. " +
          s"Automatically disabling the trigger now.")
        triggerService.disableTriggerAuto(event.trigger.getId)
        eventBus.publish(TriggerAutoDisabledEvent(event.trigger))
        notifySystemAdmins(event.trigger)
      }
    }
  }

  private def notifySystemAdmins(trigger: Trigger): Unit = {
    val emailSubject = "[Release] Trigger was disabled because of consecutive failures"
    val emailBody =
      """
        |Disabled trigger '%s' automatically due to reaching the maximum configured number of consecutive failures.
        |
        |Please fix the cause before enabling the trigger. Check [activity logs](%s#/trigger-management/%s/logs) for more details.
        |""".stripMargin.format(trigger.getTitle, ServerConfiguration.getInstance().getServerUrl, Ids.getName(trigger.getId))
    notificationService.notify(GenericSystemNotification(emailSubject, emailBody))
  }

  private def checkIfAllTriggerEventsAreFailures(countableTriggerActivityLogEntries: JList[ActivityLogEntry]): Boolean = {
    countableTriggerActivityLogEntries.stream()
      .allMatch(entry => triggerFailureEvents.contains(TriggerActivity.safeValueOf(entry.getActivityType)))
  }

  private def getCountableTriggerActivityLogEntries(triggerId: String, pageable: Pageable): JList[ActivityLogEntry] = {
    triggerActivityLogsService.getFilteredLogs(triggerId, LogsFilters.ALL, countableTriggerActivityOps, pageable)
  }
}
