package com.xebialabs.xlrelease.triggers.action

import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem
import com.xebialabs.deployit.plugin.api.udm.{Metadata, Property}
import com.xebialabs.xlrelease.actors.{ReleaseActorService, ReleaseCreated, ReleaseNotCreated}
import com.xebialabs.xlrelease.domain.Release
import com.xebialabs.xlrelease.domain.variables.Variable
import com.xebialabs.xlrelease.repository.IdType.DOMAIN
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.serialization.json.repository.ResolveOptions
import com.xebialabs.xlrelease.service.ReleaseService
import com.xebialabs.xlrelease.triggers.event_based.EventBasedTrigger
import com.xebialabs.xlrelease.variable.VariableHelper
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired

import java.util
import java.util.stream.Collectors
import scala.beans.BeanProperty
import scala.jdk.CollectionConverters._

@Metadata(virtual = false)
class CreateReleaseFromTemplateAction extends BaseConfigurationItem with TriggerAction with Logging {

  @Autowired
  @BeanProperty
  var releaseActorService: ReleaseActorService = _

  @Autowired
  @BeanProperty
  var releaseService: ReleaseService = _

  @BeanProperty
  @Property(description = "Name given to releases created by this trigger. This field can contain variables.")
  var releaseTitle: String = _

  @BeanProperty
  @Property(required = false)
  var templateId: String = _

  @BeanProperty
  @Property(asContainment = true, required = false)
  var variables: java.util.List[Variable] = new util.ArrayList[Variable]()

  @BeanProperty
  @Property(required = false)
  var tags: java.util.List[String] = new util.ArrayList[String]()

  @BeanProperty
  @Property(required = true)
  var releaseFolderId: String = _

  override def run(): TriggerActionResult = {
    tags = new util.ArrayList[String](tags)
    tags.add(Ids.getName(triggerId))
    tags.add(Ids.getName(templateId))
    // TODO: sorry, you cannot invoke API as it requires authentication which we do not have on triggers
    val result = releaseActorService.createReleaseFromTrigger(
      templateId,
      releaseFolderId,
      releaseTitle,
      variables,
      getReleaseTags,
      autoStart = true,
      scheduledStartDate = None,
      triggerId)
    result match {
      case ReleaseCreated(release) => TriggerActionExecuted(s"Created release ${getReleaseLink(release)}")
      case ReleaseNotCreated(msg) => TriggerActionSkipped(msg)
    }
  }

  override def init(trigger: EventBasedTrigger): Unit = {
    trigger.getTemplateId.foreach { templateId =>
      val template = releaseService.findById(templateId, ResolveOptions.WITH_DECORATORS)
      variables = template.getVariables.asScala
        .filter(v => v.getRequiresValue && v.getShowOnReleaseStart)
        .map { v =>
          v.setId("")
          v
        }.asJava
    }

    releaseFolderId = trigger.getReleaseFolderId.orElse(trigger.getTemplateId.map(Ids.getParentId)).orNull
  }

  def getReleaseTags: util.Set[String] = {
    new util.HashSet[String](tags)
      .stream()
      // Quick fix alert, properly convert all elements to strings, as it's possible to receive a heterogeneous list from external event during runtime
      .map[String](VariableHelper.toString)
      .collect(Collectors.toSet[String])
  }

  private def getReleaseLink(release: Release) = s"[${release.getTitle}](#/releases/${DOMAIN.convertToViewId(release.getId)})"
}
