package com.xebialabs.xlrelease.triggers.event_based

import com.fasterxml.jackson.databind.ObjectMapper
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlplatform.webhooks.domain.HttpRequestEvent
import com.xebialabs.xlplatform.webhooks.events.domain.Event
import com.xebialabs.xlplatform.webhooks.events.handlers.EventConsumerHandler
import com.xebialabs.xlrelease.domain.Trigger
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.triggers.deployment_based.StatusWebhookTrigger
import com.xebialabs.xlrelease.triggers.events.{TriggerFilterFailedEvent, TriggerFilterPassedEvent}
import com.xebialabs.xlrelease.triggers.service.TriggerService
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.context.ApplicationContext
import org.springframework.stereotype.Component

import scala.util.{Failure, Success}

@Component
class EventBasedTriggerConsumerHandler @Autowired()(triggerService: TriggerService,
                                                    applicationContext: ApplicationContext,
                                                    eventBus: XLReleaseEventBus)
  extends EventConsumerHandler[Event, EventBasedTriggerConsumer]
    with Logging
    with EventFiltering {

  override def consumerConfigType: Type = Type.valueOf(classOf[EventBasedTriggerConsumer])

  override def filter(config: EventBasedTriggerConsumer, event: Event): Boolean = {
    event match {
      case event: Event if config.enabled && event.getType.instanceOf(config._eventType) =>
        var trigger: Trigger = null
        var result: Boolean = true
        config.consumerType match {
          case t if t.instanceOf(Type.valueOf(classOf[StatusWebhookTrigger])) =>
            trigger = triggerService.findById(config.targetId).asInstanceOf[StatusWebhookTrigger]
            result = doFilter(trigger.asInstanceOf[StatusWebhookTrigger], event)
          case t if t.instanceOf(Type.valueOf(classOf[EventBasedTrigger])) =>
            trigger = triggerService.findById(config.targetId).asInstanceOf[EventBasedTrigger]
            result = doFilter(trigger.asInstanceOf[EventBasedTrigger], event, applicationContext)
        }
        try {
          eventBus.publish(TriggerFilterPassedEvent(trigger, event.getId))
          result
        } catch {
          case e: Throwable =>
            eventBus.publish(TriggerFilterFailedEvent(trigger, e.getMessage, event.getId))
            throw e
        }
      case _ => true
    }
  }

  override def consumeEvent(triggerConfig: EventBasedTriggerConsumer, event: Event): Boolean = {
    triggerService.executeSync(triggerConfig.targetId, EventBasedTriggerExecutionContext(event))
    true
  }

}

trait EventFiltering {
  self: Logging =>

  protected def doFilter(trigger: EventBasedTrigger, event: Event, applicationContext: ApplicationContext): Boolean = {
    Option(trigger.eventFilter).fold {
      logger.info(s"Executing trigger '${trigger.getTitle}' on new event")
      true
    } { eventFilter =>
      applicationContext.getAutowireCapableBeanFactory.autowireBean(eventFilter)
      eventFilter.matches(event) match {
        case Success(false) =>
          logger.debug(s"Filter not satisfied on trigger '${trigger.getTitle}' [${trigger.getId}]")
          false
        case Success(true) =>
          logger.debug(s"Filter satisfied on trigger '${trigger.getTitle}' [${trigger.getId}]")
          logger.info(s"Executing trigger '${trigger.getTitle}' on new matched event")
          true
        case Failure(exception) =>
          logger.warn(s"Error evaluating filter of trigger '${trigger.getTitle}' [${trigger.getId}]")
          throw exception
      }
    }
  }

  protected def doFilter(trigger: StatusWebhookTrigger, event: Event): Boolean = {
    val eventContentNode = new ObjectMapper().readTree(event.asInstanceOf[HttpRequestEvent].content)
    val appName = eventContentNode.get("applicationName").asText
    val env = eventContentNode.get("state").get("destination").asText
    val appversion = eventContentNode.get("state").get("versionTag").asText
    val status = eventContentNode.get("state").get("deploymentStatus").asText
    if (!isApplicationTitleMatching(trigger, appName) || !isDeploymentStatusMatching(trigger, status) ||
      !isEnvironmentTitleMatching(trigger, env) || !isAppVersionMatching(trigger, appversion)) {
      logger.debug(s"Filter not satisfied on trigger '${trigger.getTitle}' for application '$appName', environment '$env', " +
        s"version '$appversion' and status '$status'")
      false
    } else {
      logger.info(s"Executing trigger '${trigger.getTitle}' on new event for application '$appName', environment '$env', " +
        s"version '$appversion' and status '$status'")
      true
    }
  }

  private def isApplicationTitleMatching(trigger: StatusWebhookTrigger, appName: String): Boolean =
    trigger.getApplicationTitle == appName

  private def isDeploymentStatusMatching(trigger: StatusWebhookTrigger, status: String): Boolean =
    trigger.getDeploymentStatuses.contains(status.toLowerCase)

  private def isEnvironmentTitleMatching(trigger: StatusWebhookTrigger, env: String): Boolean =
    trigger.getEnvironmentTitle.isEmpty || trigger.getEnvironmentTitle == env

  private def isAppVersionMatching(trigger: StatusWebhookTrigger, appversion: String): Boolean =
    trigger.getAppVersion.isEmpty || trigger.getAppVersion == appversion || trigger.getAppVersion == "*"
}
