package com.xebialabs.xlrelease.triggers

import com.google.common.base.Strings.isNullOrEmpty
import com.xebialabs.deployit.plugin.api.validation.ExtendedValidationContext
import com.xebialabs.xlrelease.domain.Trigger
import com.xebialabs.xlrelease.repository.Ids

import scala.jdk.CollectionConverters._
import scala.util.matching.Regex

trait TriggerValidator {

  def checkTemplatePresence(extendedContext: ExtendedValidationContext, trigger: Trigger, templateId: String): Unit = {
    if (isNullOrEmpty(templateId)) {
      extendedContext.error(trigger, "templateId", s"Trigger does not have associated template")
    }
  }

  def checkTemplateFolderAndTriggerFolderMatch(extendedContext: ExtendedValidationContext, trigger: Trigger, templateId: String): Unit = {
    if (!isNullOrEmpty(trigger.getFolderId) && !isNullOrEmpty(templateId) && Ids.findFolderId(templateId) != trigger.getFolderId) {
      extendedContext.error(trigger, "templateId", s"Trigger folder id does not match template folder id")
    }
  }

  def checkReleaseTitlePresence(extendedContext: ExtendedValidationContext, trigger: Trigger, releaseTitle: String): Unit = {
    if (isNullOrEmpty(releaseTitle)) {
      extendedContext.error(trigger, "releaseTitle", "Release title is required")
    }
  }


  def checkMalformedVariableNamesInMappings(trigger: Trigger, extendedContext: ExtendedValidationContext): Unit = {
    if (trigger.isEnabled && containsMalformedVariableNames(trigger)) {
      extendedContext.focus(trigger, "mappedProperties").error(
        s"""There is an issue with the name of one or more variables.
           | Please update the variable names and save the trigger again.
           | The trigger will be disabled until the issue is fixed.""".stripMargin
      )
    }
  }


  private def containsMalformedVariableNames(trigger: Trigger): Boolean = {
    val badVarNamePatter: Regex = "variables\\[(.*(\\[|\\]).*)\\]".r
    trigger.asInstanceOf[AbstractEventTrigger].mappedProperties.asScala.exists(mappedProp => {
      badVarNamePatter.findFirstMatchIn(mappedProp.targetProperty) match {
        case Some(_) => true
        case None => false
      }
    })
  }

}
