package com.xebialabs.xlrelease.triggers.event_based

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.domain.Trigger
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.repository.{ReleaseRepository, TriggerRepository}
import com.xebialabs.xlrelease.triggers.action.TriggerAction
import com.xebialabs.xlrelease.triggers.actors.{TriggerExecutionResult, TriggerLifecycle}
import com.xebialabs.xlrelease.triggers.service.impl.TriggerExecutionContext
import com.xebialabs.xlrelease.webhooks.mapping.PropertiesMapper
import com.xebialabs.xlrelease.webhooks.registry.SubscriberRegistry
import grizzled.slf4j.Logging
import org.springframework.context.ApplicationContext

class EventBasedTriggerLifecycle(applicationContext: ApplicationContext,
                                 subscriberRegistry: SubscriberRegistry,
                                 propertiesMapper: PropertiesMapper,
                                 val eventBus: XLReleaseEventBus,
                                 val triggerRepository: TriggerRepository,
                                 val releaseRepository: ReleaseRepository
                                )
  extends TriggerLifecycle[EventBasedTrigger]
    with Logging {

  override def supports(trigger: Trigger): Boolean = trigger.getType.instanceOf(Type.valueOf(classOf[EventBasedTrigger]))

  override def enable(trigger: EventBasedTrigger, checkReferencePermissions: Boolean): Unit = {
    disable(trigger)
    subscriberRegistry.subscribe(
      sourceId = trigger.eventSource.sourceId,
      consumerConfig = new EventBasedTriggerConsumer(trigger)
    )
  }

  override def disable(trigger: EventBasedTrigger): Unit = {
    subscriberRegistry.remove(new EventBasedTriggerConsumer(trigger).consumerId)
  }

  override def execute(trigger: EventBasedTrigger, triggerExecutionContext: TriggerExecutionContext): TriggerExecutionResult[EventBasedTrigger] = {
    val result = runAction(trigger, triggerExecutionContext)
    TriggerExecutionResult(trigger, result)
  }

  override def getOrder: Int = 10

  override def getTriggerAction(trigger: EventBasedTrigger, triggerExecutionContext: TriggerExecutionContext): TriggerAction = {
    val action = trigger.instantiateAction()
    applicationContext.getAutowireCapableBeanFactory.autowireBean(action)
    action.init(trigger)
    triggerExecutionContext match {
      case EventBasedTriggerExecutionContext(event) =>
        propertiesMapper.mapProperties(trigger, event, action)
    }
    action.setTriggerId(trigger.getId)
    action
  }
}
