package com.xebialabs.xlrelease.triggers.scheduled.quartz

import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.domain.{PollType, ScheduledTrigger}
import com.xebialabs.xlrelease.quartz.release.scheduler.ReleaseSchedulerService
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.triggers.scheduled.ScheduledJobService
import com.xebialabs.xlrelease.utils.QuartzUtils._
import grizzled.slf4j.Logging
import org.quartz.CronScheduleBuilder._
import org.quartz.JobBuilder._
import org.quartz.SimpleScheduleBuilder._
import org.quartz.TriggerBuilder._
import org.quartz._
import org.springframework.retry.RetryCallback
import org.springframework.retry.support.RetryTemplateBuilder
import org.springframework.util.StringUtils.hasText

import java.util.TimeZone

class QuartzScheduledJobService(releaseSchedulerService: ReleaseSchedulerService) extends ScheduledJobService with Logging {
  val retryTemplate = new RetryTemplateBuilder().fixedBackoff(250).maxAttempts(8).build()

  import QuartzScheduledJobService._

  override def schedule(scheduledTrigger: ScheduledTrigger): Unit = {
    logger.debug(s"Schedule trigger ${scheduledTrigger.normalizedId}")
    val triggerType = scheduledTrigger.getType.toString
    val jobClass = classOf[TriggerQuartzJob]

    val job: JobDetail = newJob(jobClass)
      .withDescription(s"Trigger '${scheduledTrigger.normalizedId}'")
      .withIdentity(scheduledTrigger.normalizedId, triggerType)
      .storeDurably(true)
      .build()

    val quartzTrigger = newTrigger()
      .withIdentity(scheduledTrigger.normalizedId, triggerType)
      .withDescription(s"Trigger '${scheduledTrigger.normalizedId}'")
      .withSchedule(triggerSchedule(scheduledTrigger))
      .startNow()
      .build()

    releaseSchedulerService.schedule(job, quartzTrigger)
  }

  override def unschedule(scheduledTrigger: ScheduledTrigger): Unit = {
    logger.debug(s"Unschedule trigger ${scheduledTrigger.normalizedId}")
    val jobKey = new JobKey(scheduledTrigger.getName, scheduledTrigger.getType.toString)
    val retryCallback: RetryCallback[Boolean, SchedulerException] = _ => releaseSchedulerService.unschedule(jobKey)
    retryTemplate.execute(retryCallback)
  }

  private def triggerSchedule(releaseTrigger: ScheduledTrigger): ScheduleBuilder[_ <: Trigger] = {
    val scheduleBuilder = releaseTrigger.getPollType match {
      case PollType.REPEAT =>
        simpleSchedule()
          .withIntervalInSeconds(releaseTrigger.getPeriodicity.toInt)
          .withMisfireHandlingInstructionNextWithRemainingCount()
          .repeatForever()
      case PollType.CRON =>
        val timezoneName = XlrConfig.getInstance.quartz.timezone
        val timezone: TimeZone = if (hasText(timezoneName)) TimeZone.getTimeZone(timezoneName) else TimeZone.getDefault
        val builder = cronSchedule(releaseTrigger.getPeriodicity.asQuartzCron).inTimeZone(timezone)
        releaseTrigger.isMisfireRecoveryEnabled match {
          case true => builder.withMisfireHandlingInstructionFireAndProceed()
          case false => builder.withMisfireHandlingInstructionDoNothing()
        }
        builder
    }
    scheduleBuilder
  }
}

object QuartzScheduledJobService {

  implicit class TriggerExtension(trigger: ScheduledTrigger) {

    import com.xebialabs.xlrelease.repository.sql.persistence.CiId._

    def normalizedId: String = {
      Ids.getFolderlessId(trigger.getId.normalized)
    }
  }

}

