package com.xebialabs.xlrelease.triggers.management.repository

import com.xebialabs.deployit.security.Permissions.{authenticationToPrincipals, getAuthentication}
import com.xebialabs.deployit.security.{PermissionEnforcer, RoleService}
import com.xebialabs.xlrelease.api.v1.filter.TriggerFilters
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.repository.query.{FiltersQueryBuilder, PageableQuery, Query}
import com.xebialabs.xlrelease.repository.sql.persistence.PersistenceSupport
import com.xebialabs.xlrelease.triggers.security.TriggerPermissions._
import com.xebialabs.xlrelease.utils.ScopedTokenPermissionValidator.checkPermissionSupportedinScopedToken
import com.xebialabs.xlrelease.view.ReleaseTriggerOverview
import org.springframework.data.domain.{Page, Pageable}
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate

import scala.jdk.CollectionConverters._

class TriggerManagementRepository(val permissionEnforcer: PermissionEnforcer,
                                  val roleService: RoleService,
                                  val jdbcTemplate: JdbcTemplate,
                                  implicit val dialect: Dialect)
  extends PersistenceSupport {

  def findBy(triggerFilters: TriggerFilters, pageable: Pageable): Page[ReleaseTriggerOverview] = {
    val builder = QueryBuilder(dialect, namedTemplate).from(triggerFilters).asInstanceOf[SqlTriggerFiltersQueryBuilder]
    if (!permissionEnforcer.isCurrentUserAdmin) {
      checkPermissionSupportedinScopedToken(VIEW_TRIGGER_PERMISSION_SET: _*)
      builder.withEditableFlag(currentPrincipals(), currentRoleIds())
      builder.withPermissions(VIEW_TRIGGER_PERMISSION_SET, currentPrincipals(), currentRoleIds())
    }
    val pageableQuery: PageableQuery[ReleaseTriggerOverview] = builder.withPageable(pageable).build()
    pageableQuery.execute()
  }

  def findAllEditableTriggerIds(): java.util.List[String] = {
    val builder = TriggerListQueryBuilder(dialect, namedTemplate)
    if (!permissionEnforcer.isCurrentUserAdmin) {
      checkPermissionSupportedinScopedToken(EDIT_TRIGGER_PERMISSION_SET: _*)
      builder.withPermissions(EDIT_TRIGGER_PERMISSION_SET, currentPrincipals(), currentRoleIds())
    }
    val query: Query[java.util.List[String]] = builder.build()
    query.execute()
  }

  private def currentPrincipals(): Iterable[String] =
    authenticationToPrincipals(getAuthentication).asScala

  private def currentRoleIds(): Iterable[String] =
    roleService.getRolesFor(getAuthentication).asScala.map(_.getId)

}

trait QueryBuilder {
  def from(triggerFilters: TriggerFilters): FiltersQueryBuilder[TriggerFilters, ReleaseTriggerOverview]
}

object QueryBuilder {
  def apply(dialect: Dialect, namedTemplate: NamedParameterJdbcTemplate): QueryBuilder = new SqlQueryBuilder(dialect, namedTemplate)
}

class SqlQueryBuilder(dialect: Dialect, namedTemplate: NamedParameterJdbcTemplate) extends QueryBuilder {
  override def from(triggerFilters: TriggerFilters): FiltersQueryBuilder[TriggerFilters, ReleaseTriggerOverview] = {
    new SqlTriggerFiltersQueryBuilder(dialect, namedTemplate).from(triggerFilters)
  }
}



