package com.xebialabs.xlrelease.triggers.event_based

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlplatform.webhooks.events.domain.{DeploymentServerEvent, Event}
import com.xebialabs.xlplatform.webhooks.events.handlers.EventConsumerHandler
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.domain.Trigger
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.triggers.deployment_based.StatusWebhookTrigger
import com.xebialabs.xlrelease.triggers.events.{TriggerFilterFailedEvent, TriggerFilterPassedEvent}
import com.xebialabs.xlrelease.triggers.service.TriggerService
import com.xebialabs.xlrelease.webhooks.service.EventSourceOperationService
import grizzled.slf4j.Logging
import org.springframework.context.ApplicationContext
import org.springframework.stereotype.Component

import scala.util.{Failure, Success}

@Component
class EventBasedTriggerConsumerHandler(triggerService: TriggerService,
                                       applicationContext: ApplicationContext,
                                       operationService: EventSourceOperationService,
                                       eventBus: XLReleaseEventBus,
                                       xlrConfig: XlrConfig)
  extends EventConsumerHandler[Event, EventBasedTriggerConsumer]
    with Logging
    with EventFiltering {

  override def consumerConfigType: Type = Type.valueOf(classOf[EventBasedTriggerConsumer])

  override def filter(config: EventBasedTriggerConsumer, event: Event): Boolean = {
    if (xlrConfig.maintenanceModeEnabled) {
      logger.debug(s"[MAINTENANCE MODE] Event filtering disabled for trigger ${config.targetId}")
      return false
    }
    event match {
      case event: Event if config.enabled && event.getType.instanceOf(config._eventType) =>
        processTriggerFiltering(config, event)
      case _ => true
    }
  }

  private def processTriggerFiltering(config: EventBasedTriggerConsumer, event: Event): Boolean = {
    var trigger: Trigger = null
    var result: Boolean = true
    config.consumerType match {
      case t if t.instanceOf(Type.valueOf(classOf[StatusWebhookTrigger])) =>
        val statusWebhookTrigger = triggerService.findById(config.targetId).asInstanceOf[StatusWebhookTrigger]
        val mappedEvent = operationService.executeMapScript(statusWebhookTrigger.getEndpoint, event)
        result = doFilter(statusWebhookTrigger, mappedEvent.asInstanceOf[DeploymentServerEvent])
        trigger = statusWebhookTrigger
      case t if t.instanceOf(Type.valueOf(classOf[EventBasedTrigger])) =>
        trigger = triggerService.findById(config.targetId).asInstanceOf[EventBasedTrigger]
        result = doFilter(trigger.asInstanceOf[EventBasedTrigger], event, applicationContext)
    }
    try {
      eventBus.publish(TriggerFilterPassedEvent(trigger, event.getId))
      result
    } catch {
      case e: Throwable =>
        eventBus.publish(TriggerFilterFailedEvent(trigger, e.getMessage, event.getId))
        throw e
    }
  }

  override def consumeEvent(triggerConfig: EventBasedTriggerConsumer, event: Event): Boolean = {
    triggerService.executeSync(triggerConfig.targetId, EventBasedTriggerExecutionContext(event))
    true
  }

}

trait EventFiltering {
  self: Logging =>

  protected def doFilter(trigger: EventBasedTrigger, event: Event, applicationContext: ApplicationContext): Boolean = {
    Option(trigger.eventFilter).fold {
      logger.info(s"Executing trigger '${trigger.getTitle}' on new event")
      true
    } { eventFilter =>
      applicationContext.getAutowireCapableBeanFactory.autowireBean(eventFilter)
      eventFilter.matches(event) match {
        case Success(false) =>
          logger.debug(s"Filter not satisfied on trigger '${trigger.getTitle}' [${trigger.getId}]")
          false
        case Success(true) =>
          logger.debug(s"Filter satisfied on trigger '${trigger.getTitle}' [${trigger.getId}]")
          logger.info(s"Executing trigger '${trigger.getTitle}' on new matched event")
          true
        case Failure(exception) =>
          logger.warn(s"Error evaluating filter of trigger '${trigger.getTitle}' [${trigger.getId}]")
          throw exception
      }
    }
  }

  protected def doFilter(trigger: StatusWebhookTrigger, event: DeploymentServerEvent): Boolean = {
    logger.debug(s"Evaluating filter for trigger '${trigger.getTitle}' on event: ${event.getId}")
    if (event == null) {
      logger.warn(s"Event is null, cannot evaluate filter for trigger '${trigger.getTitle}'")
      return false
    }

    val appName = event.getApplicationTitle
    val env = event.getEnvironmentTitle
    val (version, status) = Option(event.getDeploymentState).map(state => {
      (state.getVersionTag, state.getStatus)
    }).getOrElse(("", ""))

    if (!isApplicationTitleMatching(trigger, appName) || !isDeploymentStatusMatching(trigger, status) ||
      !isEnvironmentTitleMatching(trigger, env) || !isAppVersionMatching(trigger, version)) {
      logger.debug(s"Filter not satisfied on trigger '${trigger.getTitle}' for application '$appName', environment '$env', " +
        s"version '$version' and status '$status'")
      false
    } else {
      logger.info(s"Executing trigger '${trigger.getTitle}' on new event for application '$appName', environment '$env', " +
        s"version '$version' and status '$status'")
      true
    }
  }

  private def isApplicationTitleMatching(trigger: StatusWebhookTrigger, appName: String): Boolean =
    trigger.getApplicationTitle == appName

  private def isDeploymentStatusMatching(trigger: StatusWebhookTrigger, status: String): Boolean =
    trigger.getDeploymentStatuses.contains(status.toLowerCase)

  private def isEnvironmentTitleMatching(trigger: StatusWebhookTrigger, env: String): Boolean =
    trigger.getEnvironmentTitle.isEmpty || trigger.getEnvironmentTitle == env

  private def isAppVersionMatching(trigger: StatusWebhookTrigger, appversion: String): Boolean =
    trigger.getAppVersion.isEmpty || trigger.getAppVersion == appversion || trigger.getAppVersion == "*"
}
