package com.xebialabs.xlrelease.udm.reporting.events

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.domain.events._
import com.xebialabs.xlrelease.domain.facet.Facet
import com.xebialabs.xlrelease.domain.{CustomScriptTask, Task}
import com.xebialabs.xlrelease.environments.repository.{ApplicationRepository, EnvironmentRepository}
import com.xebialabs.xlrelease.events.{EventListener, Subscribe}
import com.xebialabs.xlrelease.repository.Ids._
import com.xebialabs.xlrelease.service.{CiIdService, FacetService, FolderVariableService, VariableService}
import com.xebialabs.xlrelease.udm.reporting.DeploymentTaskFacet
import com.xebialabs.xlrelease.udm.reporting.repository.DeploymentRepository
import org.springframework.stereotype.Component

import scala.jdk.CollectionConverters._
import scala.util.Try

object XldDeploymentListener {
  val XLD_DEPLOY_TYPE = "xldeploy.Deploy"
  val PROP_DEPLOYMENT_APP = "deploymentApplication"
  val PROP_DEPLOYMENT_ENV = "deploymentEnvironment"
  val PROP_DEPLOYMENT_VERSION = "deploymentVersion"
  val PROP_DEPLOYMENT_PACKAGE = "deploymentPackage"
}

@Component
@EventListener
class XldDeploymentListener(val applicationRepository: ApplicationRepository,
                            val environmentRepository: EnvironmentRepository,
                            val deploymentRepository: DeploymentRepository,
                            val variableService: VariableService,
                            val folderVariableService: FolderVariableService,
                            val facetService: FacetService,
                            val ciIdService: CiIdService) extends DeploymentListener {

  import XldDeploymentListener._

  @Subscribe
  def onFacetEvent(event: FacetCreatedEvent): Unit = if (isDeploymentFacet(event.facet)) {
    logger.trace(s"Removing auto-generated deployment events for task ${event.facet.getTargetId}")
    deploymentRepository.deleteAutoGeneratedDeployments(event.facet.getTargetId)
  }

  @Subscribe
  def onTaskEvent(event: TaskExecutionEvent): Unit = if (isXldTaskWithoutFacet(event.task)) {
    val status = calculateDeploymentStatus(event.task)
    if (status.isDefined) {
      val facet = createOnTheFlyFacet(event.task)
      saveDeployment(event.task, facet, status.get, autoGenerated = true)
    }
  }

  private def createOnTheFlyFacet(task: Task): DeploymentTaskFacet = {
    val facetId = deploymentRepository.findByTaskId(task.getId)
      .map(_.getDeploymentId.replace("Deployment", "Facet"))
      .headOption
      .getOrElse(ciIdService.getUniqueId(Type.valueOf(classOf[Facet]), ""))

    val deploymentApplication = getName(getProperty(task, PROP_DEPLOYMENT_APP))
    val deploymentEnvironment = getName(getProperty(task, PROP_DEPLOYMENT_ENV))
    val deploymentVersion = getProperty(task, PROP_DEPLOYMENT_VERSION)
    val deploymentPackage = getProperty(task, PROP_DEPLOYMENT_PACKAGE)

    val deploymentPackageApp = Try(getName(getParentId(deploymentPackage))).getOrElse(null)
    val deploymentPackageVersion = Try(getName(deploymentPackage)).getOrElse(null)

    val facet = Type.valueOf(classOf[DeploymentTaskFacet]).getDescriptor.newInstance[DeploymentTaskFacet]("")
    facet.setId(facetId)
    facet.setTargetId(task.getId)
    facet.setApplicationId(Option(deploymentApplication).getOrElse(deploymentPackageApp))
    facet.setEnvironmentId(deploymentEnvironment)
    facet.setVersion(Option(deploymentVersion).getOrElse(deploymentPackageVersion))
    facet
  }

  private def isXldTaskWithoutFacet(task: Task): Boolean =
    isXldTask(task) && !hasDeploymentFacet(task)

  private def hasDeploymentFacet(task: Task): Boolean = {
    task.getFacets.asScala.exists(isDeploymentFacet)
  }

  private def isXldTask(task: Task): Boolean = {
    task.getTaskType.instanceOf(Type.valueOf(XLD_DEPLOY_TYPE))
  }

  private def getProperty(task: Task, property: String): String = {
    task.asInstanceOf[CustomScriptTask].getPythonScript.getProperty[String](property)
  }
}
