package com.xebialabs.xlrelease.udm.reporting.repository.sql.queries

import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.domain.status.ReleaseStatus
import com.xebialabs.xlrelease.plugins.dashboard.domain.Tile
import com.xebialabs.xlrelease.reports.filters.ReportFilter
import com.xebialabs.xlrelease.repository.ConfigurationRepository
import com.xebialabs.xlrelease.repository.sql.persistence.ReleasesSqlBuilder.alias
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.RELEASES
import com.xebialabs.xlrelease.repository.sql.persistence.{ReleasePersistence, ReleasesSqlBuilder}
import com.xebialabs.xlrelease.risk.configuration.RiskGlobalThresholds
import com.xebialabs.xlrelease.risk.configuration.RiskGlobalThresholds.RISK_GLOBAL_THRESHOLD_SETTINGS_ID
import com.xebialabs.xlrelease.udm.reporting.filters.impl.RiskFilter
import com.xebialabs.xlrelease.udm.reporting.repository.sql.queries.LiveReleasesByRiskQuery.{ATTENTION_NEEDED, AT_RISK, ON_TRACK}
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.stereotype.Component

import java.sql.ResultSet
import java.util.{ArrayList => JArrayList, List => JList, Map => JMap}
import scala.jdk.CollectionConverters._

@Component
@IsTransactional
class LiveReleasesByRiskQuery @Autowired()(configurationRepository: ConfigurationRepository,
                                           releasePersistence: ReleasePersistence,
                                           implicit val detailsQuery: LiveReleasesQuery,
                                           @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect) extends LiveQuery {

  private val RELEASE_COUNT = "releaseCount"
  private val RELEASE_COLUMNS = s"$alias.${RELEASES.TOTAL_RISK_SCORE}, COUNT($alias.${RELEASES.CI_UID}) ${RELEASE_COUNT}"

  private val DEFAULT_STATUS_FILTER = statusFilter(ReleaseStatus.ACTIVE_STATUSES :+ ReleaseStatus.PLANNED)

  @IsReadOnly
  override def executeSummary(tile: Tile, additionalParameters: JMap[String, Any]): AnyRef = {
    getRiskCounts(tile)
  }

  @IsReadOnly
  override def executeDetails(tile: Tile, additionalParameters: JMap[String, Any]): AnyRef = {
    val riskLevel = additionalParameters.getOrDefault("view", AT_RISK).asInstanceOf[String]

    val filters = getFilters(tile, riskLevel)
    tile.setProperty("filters", filters)

    detailsQuery.executeDetails(tile, additionalParameters)
  }

  private def getFilters(tile: Tile, riskLevel: String): JList[ReportFilter] = {
    val filters = new JArrayList[ReportFilter]
    filters.addAll(tile.getProperty[JList[ReportFilter]]("filters"))
    filters.add(DEFAULT_STATUS_FILTER)

    if (riskLevel != null) {
      val thresholds = configurationRepository.read(RISK_GLOBAL_THRESHOLD_SETTINGS_ID).asInstanceOf[RiskGlobalThresholds]

      filters.add(riskLevel match {
        case ON_TRACK => new RiskFilter(0, thresholds.getAttentionNeededFrom - 1)
        case ATTENTION_NEEDED => new RiskFilter(thresholds.getAttentionNeededFrom, thresholds.getAtRiskFrom - 1)
        case AT_RISK => new RiskFilter(thresholds.getAtRiskFrom, -1)
      })
    }
    filters
  }

  private def getRiskCounts(tile: Tile): AnyRef = {
    val filters = getFilters(tile, null)

    val builder = new ReleasesSqlBuilder()
      .customSelect(RELEASE_COLUMNS)
      .withFilters(filters)
      .groupBy(s"$alias.${RELEASES.TOTAL_RISK_SCORE}")

    val (sql, params) = builder.build()
    val counts = releasePersistence.jdbcTemplate.query[RiskCount](sql, params.toArray, (rs: ResultSet, _: Int) => {
      RiskCount(rs.getInt(RELEASES.TOTAL_RISK_SCORE), rs.getInt(RELEASE_COUNT))
    }).asScala

    val thresholds = configurationRepository.read[RiskGlobalThresholds](RISK_GLOBAL_THRESHOLD_SETTINGS_ID)

    val result = counts.groupBy(riskCount => if (riskCount.score < thresholds.getAttentionNeededFrom) {
      ON_TRACK
    } else if (riskCount.score < thresholds.getAtRiskFrom) {
      ATTENTION_NEEDED
    } else {
      AT_RISK
    }).view.mapValues(counts => counts.map(_.count).sum).toMap

    Map(
      ON_TRACK -> result.getOrElse(ON_TRACK, 0),
      ATTENTION_NEEDED -> result.getOrElse(ATTENTION_NEEDED, 0),
      AT_RISK -> result.getOrElse(AT_RISK, 0)
    ).asJava
  }
}

object LiveReleasesByRiskQuery {
  val ON_TRACK = "on_track"
  val ATTENTION_NEEDED = "attention_needed"
  val AT_RISK = "at_risk"
}

case class RiskCount(score: Int, count: Int)
