package com.xebialabs.xlrelease.webhooks.registry

import akka.actor.{Actor, ActorLogging, ActorRef, FSM, Props}
import akka.cluster.pubsub.DistributedPubSub
import akka.cluster.pubsub.DistributedPubSubMediator.{Publish, Subscribe, SubscribeAck}
import com.xebialabs.xlplatform.webhooks.events.domain.EventConsumer.CiEventConsumer
import com.xebialabs.xlplatform.webhooks.events.domain.{EventConsumer, EventSource}
import com.xebialabs.xlrelease.webhooks.registry.SubscriberRegistryActor._

object SubscriberRegistryActor {
  final val TOPIC: String = "subscriberRegistry"

  def props(subscriberRegistry: ClusterSubscriberRegistryService) = Props(new SubscriberRegistryActor(subscriberRegistry))

  sealed trait SubscriberRegistryMessage
  sealed trait SubscriberRegistryCommand extends SubscriberRegistryMessage
  case class SubscribeConsumer(sourceId: EventSource.ID, consumerConfig: CiEventConsumer) extends SubscriberRegistryCommand
  case class UnsubscribeConsumer(sourceId: EventSource.ID, consumerId: EventConsumer.ID) extends SubscriberRegistryCommand
  case class RemoveConsumer(consumerId: EventConsumer.ID) extends SubscriberRegistryCommand
  case class Broadcast(command: SubscriberRegistryCommand) extends SubscriberRegistryMessage

  sealed trait SubscriberRegistryActorState
  case object Running extends SubscriberRegistryActorState

}

class SubscriberRegistryActor(subscriberRegistry: ClusterSubscriberRegistryService)
  extends Actor
    with FSM[SubscriberRegistryActorState, Unit]
    with ActorLogging {

  private val mediator: ActorRef = DistributedPubSub(context.system).mediator

  override def preStart(): Unit = {
    super.preStart()
    log.debug(s"subscribing to $TOPIC")
    mediator ! Subscribe(TOPIC, self)
  }

  startWith(Running, ())

  when(Running) {
    case Event(SubscribeAck(Subscribe(TOPIC, None, `self`)), _) =>
      log.debug(s"subscribed to $TOPIC")
      stay()
    case Event(Broadcast(command), _) =>
      log.debug(s">> $command to remote nodes")
      mediator ! Publish(TOPIC, command)
      stay()
    case Event(SubscribeConsumer(sourceId, consumer), _) =>
      log.debug(s"<< subscribed consumer from remote: $sourceId -> $consumer")
      subscriberRegistry.remoteSubscribe(sourceId, consumer)
      stay()
    case Event(UnsubscribeConsumer(sourceId, consumerId), _) =>
      log.debug(s"<< unsubscribed consumer from remote: $sourceId -> $consumerId")
      subscriberRegistry.remoteUnsubscribe(sourceId, consumerId)
      stay()
    case Event(RemoveConsumer(consumerId), _) =>
      log.debug(s"removed consumer from remote: $consumerId")
      subscriberRegistry.remoteRemove(consumerId)
      stay()
    case anything =>
      log.warning(s"<< What's this? $anything")
      stay()
  }

  whenUnhandled {
    case Event(msg, _) =>
      log.warning(s"<< Unhandled: $msg")
      stay()
  }

  initialize()
}
