package com.xebialabs.xlplatform.webhooks.endpoint

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlplatform.webhooks.authentication.RequestAuthenticationMethod
import com.xebialabs.xlplatform.webhooks.domain.{Endpoint, HttpRequestEvent}
import com.xebialabs.xlplatform.webhooks.endpoint.exceptions.EndpointNotFound
import com.xebialabs.xlplatform.webhooks.events.handlers.EventSourceHandler
import grizzled.slf4j.Logging
import org.apache.commons.io.IOUtils

import javax.servlet.http.HttpServletRequest
import javax.ws.rs.core.{Context, Response}
import javax.ws.rs.{GET, POST, Path, PathParam}
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Try}


abstract class WebhooksEndpointController(endpointProvider: EndpointProvider)
  extends EventSourceHandler[HttpRequestEvent, Endpoint]
    with AuthenticationMethodSupport
    with RequestAcceptor
    with EndpointValidator
    with Logging {

  @GET
  @Path("{endpointPath}")
  def acceptGET(@PathParam("endpointPath") path: String,
                @Context request: HttpServletRequest): Response = {
    accept(path, request)
  }

  @POST
  @Path("{endpointPath}")
  def acceptPOST(@PathParam("endpointPath") path: String,
                 @Context request: HttpServletRequest): Response = {
    accept(path, request)
  }



  protected override def processRequest(path: String, request: HttpServletRequest): Try[Response] = {
    def getHeaders = request.getHeaderNames.asScala.map(name => name -> request.getHeader(name)).toMap

    def getParams = request.getParameterMap.asScala.toMap

    for {
      endpoint <- getEndpoint(path)
      _ <- checkEnabled(endpoint)
      _ <- checkMethod(request, endpoint)
      payload <- Try(IOUtils.toString(request.getReader))
      reqHeaders = getHeaders
      reqParams = getParams
      authenticationMethod <- getRequestAuthenticationMethod(endpoint)
      _ <- authenticateRequest(endpoint, reqHeaders, reqParams, payload)(authenticationMethod)
      event = HttpRequestEvent(endpoint, reqHeaders.asJava, reqParams.asJava, payload)
      published <- Try(publish(endpoint, event))
      response = if (published) Response.ok() else Response.notAcceptable(List.empty.asJava)
    } yield {
      logger.trace(s"${requestPrefix(request)}: Published event for $endpoint, authenticated with ${authenticationMethod.getClass.getName}")
      response.build()
    }
  }

  protected def getEndpoint(path: String): Try[Endpoint] = {
    endpointProvider.findEndpointByPath(path)
      .recoverWith {
        case _: NotFoundException => Failure(EndpointNotFound(path))
      }
  }
}

// TODO: we should really not pollute the logs with stack traces when rejecting an incoming payload.
object exceptions {

  abstract class WebhookEndpointControllerException(val message: String,
                                                    val status: Response.Status,
                                                    val logStackTrace: Boolean = false)
    extends Throwable(message)

  case class EndpointNotFound(path: String)
    extends WebhookEndpointControllerException(
      message = s"Endpoint not found for path '$path'",
      status = Response.Status.NOT_FOUND
    )

  case class WrongMethod(req: HttpServletRequest, endpoint: Endpoint)
    extends WebhookEndpointControllerException(
      message = s"Wrong HTTP method for '$endpoint': expected ${endpoint.method}, got ${req.getMethod}.",
      status = Response.Status.BAD_REQUEST
    )

  case class EndpointAuthenticationMethodNotFound(endpoint: Endpoint)
    extends WebhookEndpointControllerException(
      message = s"Authentication method not found for '$endpoint'.",
      status = Response.Status.NOT_FOUND
    )

  case class UnauthorizedRequest(endpoint: Endpoint, authenticationMethod: RequestAuthenticationMethod)
    extends WebhookEndpointControllerException(
      message = s"Unauthorized request for '$endpoint' (authentication method: ${authenticationMethod.getClass.getName})",
      status = Response.Status.UNAUTHORIZED
    )

  case class EndpointDisabled(endpoint: Endpoint)
    extends WebhookEndpointControllerException(
      message = s"Endpoint '$endpoint' is disabled",
      status = Response.Status.NOT_FOUND
    )

}
