package com.xebialabs.xlrelease.webhooks.configuration.jms

import org.apache.activemq.ActiveMQConnectionFactory
import org.apache.activemq.broker.SslContext
import org.apache.activemq.command.ActiveMQQueue
import org.apache.activemq.transport.Transport
import org.apache.activemq.util.JMSExceptionSupport
import org.springframework.web.util.DefaultUriBuilderFactory

import jakarta.jms.{ConnectionFactory, Destination}

class ActiveMqWebhookJmsConfiguration(config: ActiveMqWebhookQueueConfig)
  extends BasicWebhookJmsConfiguration {

  override def jmsConnectionFactory: ConnectionFactory = {
    // see https://activemq.apache.org/components/classic/documentation/ssl-transport-reference
    // in theory we could use ActiveMQSslConnectionFactory but then we would not go via SslContextFactory
    val sslContextConfig = SslContextConfig.fromConfig(config.sslConfig)
    val activeMQConnectionFactory = new ActiveMQConnectionFactory() {
      override def createTransport(): Transport = {
        val existing = SslContext.getCurrentSslContext
        try {
          if (config.useSsl) {
            val javaSslContext = SslContextFactory.getContext(sslContextConfig)
            val activeMqSslContext = new SslContext()
            activeMqSslContext.setSSLContext(javaSslContext)
            SslContext.setCurrentSslContext(activeMqSslContext)
          }
          super.createTransport()
        } catch {
          case e: Exception => throw JMSExceptionSupport.create(s"Could not create Transport. Reason: ${e.getMessage}", e)
        } finally {
          SslContext.setCurrentSslContext(existing)
        }
      }
    }
    val brokerUriBuilder = new DefaultUriBuilderFactory(config.url).builder()
    if (config.useSsl) {
      brokerUriBuilder.queryParam("verifyHostName", sslContextConfig.hostnameVerification)
    }
    val brokerUrl = brokerUriBuilder.build().toString
    activeMQConnectionFactory.setBrokerURL(brokerUrl)
    activeMQConnectionFactory.setUserName(config.username)
    activeMQConnectionFactory.setPassword(config.password)
    activeMQConnectionFactory.setTrustAllPackages(true)
    activeMQConnectionFactory
  }

  override def webhookEventQueue: Destination = {
    new ActiveMQQueue(config.queueName)
  }

}
