package com.xebialabs.xlrelease.webhooks.registry

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlplatform.webhooks.domain.Endpoint
import com.xebialabs.xlplatform.webhooks.endpoint.EndpointProvider
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.repository.sql.SqlRepositoryAdapter
import com.xebialabs.xlrelease.repository.sql.persistence.PersistenceSupport
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.CONFIGURATIONS
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import com.xebialabs.xlrelease.webhooks.repository.persistence.WebhookSchema.WEBHOOK_ENDPOINTS
import org.springframework.jdbc.core.JdbcTemplate

import scala.util.{Failure, Success, Try}

object EndpointRegistry
  extends EndpointRegistry
    with EndpointProvider {

  private var instance: EndpointRegistry = _

  def apply(jdbcTemplate: JdbcTemplate, dialect: Dialect, sqlRepositoryAdapter: SqlRepositoryAdapter): EndpointRegistry = {
    instance = new EndpointRegistryImpl(jdbcTemplate, dialect, sqlRepositoryAdapter)
    instance
  }

  override def findEndpoint(path: String): Option[Endpoint] = {
    assertInstanceIsNotNull()
    instance.findEndpoint(path)
  }

  override def findEndpointByPath(path: String): Try[Endpoint] = {
    assertInstanceIsNotNull()
    instance.findEndpointByPath(path)
  }

  private def assertInstanceIsNotNull(): Unit = {
    if (null == instance) {
      throw new IllegalStateException("EndpointRegistry is not initialized")
    }
  }
}

private class EndpointRegistryImpl(val jdbcTemplate: JdbcTemplate, val dialect: Dialect, sqlRepositoryAdapter: SqlRepositoryAdapter)
  extends EndpointRegistry
    with PersistenceSupport {
  override def findEndpoint(path: String): Option[Endpoint] = {
    val STMT_FIND_BY_PATH =
      s"""SELECT
         |  conf.${CONFIGURATIONS.ID}
         | FROM ${CONFIGURATIONS.TABLE} conf
         | JOIN ${WEBHOOK_ENDPOINTS.TABLE} e ON conf.${CONFIGURATIONS.CI_UID} = e.${WEBHOOK_ENDPOINTS.CONFIGURATION_UID}
         | WHERE e.${WEBHOOK_ENDPOINTS.PATH} = :${WEBHOOK_ENDPOINTS.PATH}
         |""".stripMargin
    sqlQuery(STMT_FIND_BY_PATH, params(WEBHOOK_ENDPOINTS.PATH -> path), (rs, _) => rs.getString(CONFIGURATIONS.ID))
      .headOption match {
      case Some(ciID) => Option(sqlRepositoryAdapter.read(ciID))
      case _ => None
    }
  }

  override def findEndpointByPath(path: String): Try[Endpoint] = {
    findEndpoint(path).map(Success(_)).getOrElse(Failure(new NotFoundException(s"The configuration with path='$path' does not exist")))
  }
}

trait EndpointRegistry extends EndpointProvider {
  def findEndpoint(path: String): Option[Endpoint]
}
