package com.xebialabs.xlrelease.webhooks.service.script

import com.fasterxml.jackson.annotation.JsonProperty
import com.xebialabs.deployit.engine.spi.exception.DeployitException
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.domain.Configuration
import com.xebialabs.xlrelease.script.XlrScriptContext
import com.xebialabs.xlrelease.script.builder.XlrScripContextExtensions
import com.xebialabs.xlrelease.service.AutoconfigExecutor
import com.xebialabs.xlrelease.views.AutoconfigResponse
import com.xebialabs.xlrelease.webhooks.consumers.StatusWebhookEventSource
import com.xebialabs.xlrelease.webhooks.service.EventSourceOperationService
import grizzled.slf4j.Logging
import org.springframework.stereotype.Component

import scala.util.{Failure, Success, Try}

@Component
class StatusWebhookAutoconfigureExecutor(operationService: EventSourceOperationService) extends AutoconfigExecutor with Logging {
  override def supports(configuration: Configuration): Boolean =
    configuration.getType.instanceOf(Type.valueOf(classOf[StatusWebhookEventSource]))

  override def autoconfigure(configuration: Configuration): AutoconfigResponse = {
    configuration match {
      case webhook: StatusWebhookEventSource =>
        Try(operationService.executeAutoconfigureScript(webhook)) match {
          case Success(result: AutoconfigResult) =>
            new AutoconfigResponse(true, result.message, result.details, result.url)
          case Success(errorObject) =>
            error(s"Illegal response in ${webhook.getId} autoconfiguration response." +
              s" Returned type was ${errorObject.getClass.getName}, expected autoconfig result")
            new AutoconfigResponse(false, "Could not resolve script result. Check logs for more details.", null, null)
          case Failure(exception) =>
            val msg = s"Error executing autoconfiguration on ${webhook.getId} " +
              s"[${webhook.getTitle}]: ${exception.getMessage}"
            warn(msg, exception)
            throw AutoconfigureConfigurationError(formatErrorMessage(msg))
        }
      case _ =>
        val msg = s"Cannot execute autoconfiguration on type: ${configuration.getType.toString}"
        warn(msg)
        throw AutoconfigureConfigurationError(msg)
    }
  }

}

object StatusWebhookAutoconfigureExecutor {
  def createAutoconfigScriptContext(webhook: StatusWebhookEventSource): XlrScriptContext =
    new AutoconfigScriptContextBuilder(webhook).build()
}

private class AutoconfigScriptContextBuilder(
                                              webhook: StatusWebhookEventSource
                                            ) extends BaseScriptContextBuilder(webhook) {

  override val scriptProperty: String = AutoconfigScriptProperty

  override protected def doBuild(context: XlrScriptContext): Unit = {
    addConnectionServerAttributes(context)
    context.addCiAsMap("webhook", webhook)
    addScript(context)
  }
}

case class AutoconfigResult(
                             @JsonProperty("message") message: String,
                             @JsonProperty("details") details: String,
                             @JsonProperty("url") url: String
                           )

case class AutoconfigureConfigurationError(msg: String) extends DeployitException(msg)
