angular.module('xlrelease').factory('Authenticator', ['$http', 'Backend', '$rootScope', '$window', 'Base64', '$location', function ($http, Backend, $rootScope, $window, Base64, $location) {
    var ADMINISTRATOR = 'admin';
    var START_PAGE = '/tasks';
    var FORBIDDEN_CONTENT_PAGE = '/forbidden-access';

    var AUTHENTICATION_DATA = 'authentication_data';
    var PERMISSIONS = 'permissions';

    var localStorage = $window.localStorage;

    function getAuthenticationData() {
        var authenticationData = localStorage.getItem(AUTHENTICATION_DATA);
        return authenticationData !== null ? JSON.parse(authenticationData) : null;
    }

    function setCredentials(username, password) {
        var credentials = Base64.getBasicAuthorization(username, password);
        $http.defaults.headers.common.Authorization = credentials;

        var authenticationData = {
            username: username,
            credentials: credentials
        };

        localStorage.setItem(AUTHENTICATION_DATA, JSON.stringify(authenticationData));
    }

    function getPermissions() {
        var permissions = localStorage.getItem(PERMISSIONS);
        return permissions !== null ? JSON.parse(permissions) : null;
    }

    function setPermissions(permissions) {
        localStorage.setItem(PERMISSIONS, JSON.stringify(permissions));
    }

    $rootScope.$on('UNAUTHORIZED', function () {
        authenticationService.logout();
    });

    $rootScope.$on('FORBIDDEN', function (event, errorMessage) {
        $location.path(FORBIDDEN_CONTENT_PAGE + '/' + encodeURIComponent(errorMessage));
    });

    $rootScope.$on('REFRESH_PERMISSIONS', function() {
        Backend.get('login').success(setPermissions);
    });

    function isAdmin(permissions) {
        return authenticationService.getUsername() === ADMINISTRATOR || _.contains(permissions.global, ADMINISTRATOR);
    }

    function hasAppropriatePermission(release, templatePermission, releasePermission) {
        if (release && release.status === 'TEMPLATE') {
            return authenticationService.hasPermission(templatePermission, release.id);
        }
        return release && authenticationService.hasPermission(releasePermission, release.id);
    }

    var authenticationService = {
        init: function () {
            var authenticationData = getAuthenticationData();
            if (authenticationData !== null) {
                $http.defaults.headers.common.Authorization = authenticationData.credentials;
            }

            $rootScope.security = {
                isAuthenticated: authenticationService.isAuthenticated,
                getUsername: authenticationService.getUsername,
                hasPermission: authenticationService.hasPermission,
                hasViewPermission: authenticationService.hasViewPermission,
                hasEditPermission: authenticationService.hasEditPermission,
                hasEditTaskPermission: authenticationService.hasEditTaskPermission,
                hasReassignTaskPermission: authenticationService.hasReassignTaskPermission,
                isOwner: authenticationService.isOwner
            };
        },
        getUsername: function () {
            var authenticationData = getAuthenticationData();
            return authenticationData !== null ? authenticationData.username : null;
        },
        isAuthenticated: function () {
            return getAuthenticationData() !== null;
        },
        hasPermission: function (permission, entityId) {
            var permissions = getPermissions();
            if (permissions === null) {
                return false;
            }
            if (isAdmin(permissions)) {
                return true;
            }

            if (angular.isUndefined(entityId)) {
                return _.contains(permissions.global, permission);
            } else {
                return _.contains(permissions["Applications/" + entityId], permission);
            }
        },
        hasViewPermission: function (release) {
            return hasAppropriatePermission(release, "template#view", "release#view");
        },
        hasEditPermission: function (release) {
            return hasAppropriatePermission(release, "template#edit", "release#edit");
        },
        hasEditTaskPermission: function (release) {
            return hasAppropriatePermission(release, "template#edit", "release#edit_task");
        },
        hasReassignTaskPermission: function (release) {
            return hasAppropriatePermission(release, "template#edit", "release#reassign_task");
        },
        isOwner: function (objectOwner) {
            var permissions = getPermissions();
            return permissions != null && (this.getUsername() === objectOwner || isAdmin(permissions));
        },
        login: function (username, password) {
            return Backend.get('login', {
                headers: {
                    'Authorization': Base64.getBasicAuthorization(username, password)
                }
            }).success(function(permissions) {
                setCredentials(username, password);
                setPermissions(permissions);
                $location.path(START_PAGE);
            });
        },
        logout: function () {
            delete $http.defaults.headers.common.Authorization;
            localStorage.removeItem(AUTHENTICATION_DATA);
            localStorage.removeItem(PERMISSIONS);
            $location.path('/login');
        }
    };

    authenticationService.init();
    return authenticationService;
}]);
