'use strict';

angular.module('xlrelease').factory('Calendar', ['DateService', 'ReleasesService', function (DateService, ReleasesService) {

    var MINIMUM_MONTHS = 12;
    var EXTRA_MONTHS = 5;
    var TOTAL_WEEKS_DISPLAYED = 6;
    var URL_DATE_FORMAT = "MM-YYYY";

    var calendar = {
        init: function (releases, date) {
            return {
                weeks: getWeeks(date, releases),
                months: getMonths(releases)
            };
        },
        currentMonth: function () {
            return DateService.getToday().date(1).format(URL_DATE_FORMAT);
        },
        getNumberOfDaysAfter: function (startDate, firstDayOfWeek) {
            var diff = startDate.diff(firstDayOfWeek, 'days', true);
            return diff < 0 ? 0 : diff;
        },
        getDuration: function (startDate, endDate, firstDay, lastDay) {
            return lastDay.diff(firstDay, 'days') - this.getNumberOfDaysAfter(startDate, firstDay) - this.getNumberOfDaysAfter(lastDay, endDate);
        },
        isActiveDuringWeek: function (startDate, dueDate, firstDayOfWeek) {
            var lastDayOfWeek = moment(firstDayOfWeek).add('days', 7);
            return startDate && dueDate && firstDayOfWeek.isBefore(dueDate) && lastDayOfWeek.isAfter(startDate);
        }
    };

    function getWeeks(date, releases) {
        var weeks = [];
        var day = moment(date).day(0); // sunday of current week

        for (var weekIndex = 0; weekIndex < TOTAL_WEEKS_DISPLAYED; weekIndex++) {
            var firstDayOfWeek = DateService.atMidnight(moment(day).day(0));
            var lastDayOfWeek = moment(firstDayOfWeek).add('days', 7);

            weeks.push({
                days: getDaysOfTheWeek(firstDayOfWeek),
                releases: getReleasesOfTheWeek(releases, firstDayOfWeek, lastDayOfWeek)
            });

            day.add('days', 7);
        }

        return weeks;
    }

    function getListOfMonths(numberOfMonths, firstDayOfCalendar) {
        var months = [];
        var firstDayOfMonth = moment(firstDayOfCalendar);
        for (var i = 0; i < numberOfMonths; i++) {
            months.push({
                title: firstDayOfMonth.format("MMMM"),
                year: firstDayOfMonth.format("YYYY"),
                index: firstDayOfMonth.month(),
                fullDate: moment(firstDayOfMonth)
            });

            firstDayOfMonth.add('months', 1);
        }

        return months;
    }

    function getReleasesDate(releases, finder, onProperty) {
        if (releases.length > 0) {
            var release = finder(releases, function (release) {
                if (release[onProperty]) {
                    return moment(release[onProperty]).valueOf()
                }
            });
            return moment(release[onProperty]);
        } else {
            return DateService.getToday();
        }
    }

    function getMonths(releases) {
        var firstDayOfReleases = getReleasesDate(releases, _.min, 'scheduledStartDate');
        var lastDayOfReleases = getReleasesDate(releases, _.max, 'dueDate');

        var numberOfMonths = lastDayOfReleases.diff(firstDayOfReleases, 'months') + EXTRA_MONTHS;
        numberOfMonths = numberOfMonths < MINIMUM_MONTHS ? MINIMUM_MONTHS : numberOfMonths;

        var firstDayOfCalendar = moment(firstDayOfReleases).date(1).subtract('months', Math.floor(EXTRA_MONTHS / 2));

        return getListOfMonths(numberOfMonths, firstDayOfCalendar);
    }

    function getDisplayablePhases(release, firstDayOfWeek, lastDayOfWeek) {
        var phases = release.phases;
        var phasesOfTheWeek = [];
        _.each(phases, function (phase) {
            if (phase.scheduledStartDate !== null && phase.dueDate !== null) {
                var dates = ReleasesService.cropToReleaseDate(phase, release);
                var startDate = dates.startDate;
                var dueDate = dates.dueDate;

                if (calendar.isActiveDuringWeek(startDate, dueDate, firstDayOfWeek) && isInsideRelease(release, phase)) {
                    phase.leftDayOffset = calendar.getNumberOfDaysAfter(startDate, firstDayOfWeek);
                    phase.numberOfDay = calendar.getDuration(startDate, dueDate, firstDayOfWeek, lastDayOfWeek);
                    phasesOfTheWeek.push(phase);
                }
            }
        });
        return phasesOfTheWeek;
    }

    function isInsideRelease(release, phase) {
        return moment(phase.scheduledStartDate).isBefore(moment(release.dueDate)) && moment(phase.dueDate).isAfter(moment(release.scheduledStartDate));
    }

    function getReleasesOfTheWeek(releases, firstDayOfWeek, lastDayOfWeek) {
        var releasesOfTheWeek = [];

        _.each(releases, function (release) {
            var dueDate = moment(release.dueDate);
            var startDate = moment(release.scheduledStartDate);

            if (calendar.isActiveDuringWeek(startDate, dueDate, firstDayOfWeek)) {
                var enhancedRelease = angular.copy(release);

                angular.extend(enhancedRelease, {
                    currentTask: release.currentTask ? release.currentTask.title : undefined,
                    leftDayOffset: calendar.getNumberOfDaysAfter(startDate, firstDayOfWeek),
                    numberOfDay: calendar.getDuration(startDate, dueDate, firstDayOfWeek, lastDayOfWeek),
                    displayablePhases: getDisplayablePhases(enhancedRelease, firstDayOfWeek, lastDayOfWeek)
                });

                releasesOfTheWeek.push(enhancedRelease);
            }
        });
        return releasesOfTheWeek;
    }

    function getDaysOfTheWeek(firstDayOfWeek) {
        var dayInWeek = moment(firstDayOfWeek);
        var days = [];
        for (var dayIndex = 0; dayIndex < 7; dayIndex++) {
            days.push({
                date: dayInWeek.date(),
                month: dayInWeek.format("MMMM"),
                dateStatus: DateService.getDateStatus(dayInWeek),
                fullDate: moment(dayInWeek),
                year: dayInWeek.format("YYYY")
            });
            dayInWeek.add('days', 1);
        }
        return days;
    }

    return calendar
}]);
