angular.module('xlrelease').factory('inlineEditor', ['Browser', function (Browser) {
    function leaveEditMode(scope, editable) {
        scope.editMode = false;
        editable.parents(".modal").focus();
    }

    function canCommit(scope) {
        return !scope.isRequired
            || (angular.isDefined(scope.draft)
                && scope.draft !== null
                && scope.draft !== '');
    }

    function commit(scope, editable) {
        if (canCommit(scope)) {
            scope.$apply(function () {
                scope.model = scope.draft;
                leaveEditMode(scope, editable);
            });
            // Callback must be called in a separate apply call in order to update the text property first.
            scope.$apply(function () {
                scope.onChange();
            });
        }
    }

    function rollback(scope, editable) {
        scope.$apply(function () {
            scope.draft = scope.model;
            leaveEditMode(scope, editable);
        });
    }

    function edit(scope, editable) {
        scope.$apply(function () {
            scope.editMode = true;
            scope.draft = scope.model;
        });

        if (editable.is('select')) {
            editable.focus();
        } else {
            editable.select();
        }
    }

    function commitOnBlur(scope, editable) {
        editable.on('blur', function () {
            if (scope.editMode) {
                commit(scope, editable);
            }
        });
    }

    function setupAutocomplete(scope, editable) {
        scope.$watch('autocompleteData', function (newData) {
            if (newData) {
                editable.autocomplete({
                    autoFocus: true,
                    source: newData,
                    delay: 0,
                    minLength: 0,
                    position: {
                        my: "left top",
                        at: "left bottom",
                        collision: "flip",
                        within: ".modal-body"
                    },
                    // when autocomplete menu is open, we temporarily disable 'save on blur' feature in IE8
                    open: function () {
                        if (Browser.isIE8) {
                            editable.unbind('blur');
                        }
                    },
                    close: function () {
                        editable.trigger("input");
                        if (Browser.isIE8) {
                            commitOnBlur(scope, editable);
                        }
                    },
                    select: function (event, ui) {
                        scope.$apply(function () {
                            scope.draft = ui.item.value;
                        });
                        commit(scope, editable);
                    }
                });
            }
        });
    }

    function setupRequired(scope, editable, attrs) {
        scope.isRequired = angular.isDefined(attrs.required);
        if (scope.isRequired) {
            editable.attr('required', 'required');
            editable.attr('placeholder', 'required');
        }
    }

    return {
        directive: function (template, multiline) {
            return {
                restrict: 'A',
                templateUrl: 'partials/inline-editor/' + template,
                scope: {
                    placeholder: '@',
                    model: '=',
                    onChange: '&',
                    selectOptions: '=',
                    autocompleteData: '=',
                    helpUrl: '@'
                },
                transclude: true,
                link: function (scope, element, attrs) {
                    scope.editMode = false;
                    var editable = element.find('.editable');

                    // On click
                    element.find('.display').on('click', function () {
                        edit(scope, editable);
                    });

                    if (multiline) {
                        element.find('.ok').on('click', function () {
                            commit(scope, editable);
                        });
                        element.find('.cancel').on('click', function () {
                            rollback(scope, editable);
                        });
                    } else {
                        // On enter
                        editable.on('keypress', function (e) {
                            if (e.which === 13 && e.ctrlKey === false) {
                                commit(scope, editable);
                            }
                        });
                    }

                    setupAutocomplete(scope, editable);
                    setupRequired(scope, editable, attrs);

                    // On blur
                    if (!multiline) {
                        commitOnBlur(scope, editable);
                    }

                    // On escape
                    editable.on('keyup', function (e) {
                        if (e.which === 27) {
                            rollback(scope, editable);
                            e.stopPropagation();
                        }
                    });


                    scope.isNumber = angular.isNumber;
                    scope.hasText = function (data) {
                        return ! _.isEmpty(data);
                    };
                }
            };
        }
    };
}]);

angular.module('xlrelease').directive('inlineTextEditor', function (inlineEditor) {
    return inlineEditor.directive('text-field.html', false);
});

angular.module('xlrelease').directive('inlinePasswordEditor', function (inlineEditor) {
    return inlineEditor.directive('password-field.html', false);
});

angular.module('xlrelease').directive('inlineNumberEditor', function (inlineEditor) {
    return inlineEditor.directive('number-field.html', false);
});

angular.module('xlrelease').directive('inlineTextareaEditor', function (inlineEditor) {
    return inlineEditor.directive('textarea.html', true);
});

angular.module('xlrelease').directive('inlineSelectEditor', function (inlineEditor) {
    return inlineEditor.directive('select.html', false);
});

angular.module('xlrelease').directive('inlineListEditor', function (inlineEditor) {
    return inlineEditor.directive('list.html', false);
});
