/**
 * @description
 * Directive to manage tag list input with auto-completion.
 *
 * @example
 * <div tags-input tags="releaseForm.tags" placeholder="Add a tag..." completion-candidates="candidates"></div>
 *
 * @param tags List of values within the scope.
 * @param placeholder Placeholder for the input.
 * @param completion-candidates List of candidates values within the scope.
 * @param readOnly Read only mode.
 * @param on-change Callback when tags is updated.
 */
angular.module('xlrelease').directive('tagsInput', function () {
    return {
        templateUrl: "partials/directives/tags-input.html",
        scope: {
            tags: '=',
            readOnly: '=',
            completionCandidates: '=',
            placeholder: '@',
            onChange: '&'
        },
        controller: ['$scope', '$element', function (scope, element) {
            var ctrl = this;

            ctrl.add = function (tag) {
                if (tag !== '') {
                    var alreadyExist;

                    scope.$apply(function () {
                        alreadyExist = _.contains(scope.tags, tag);
                        if (!alreadyExist) {
                            scope.tags.push(tag);
                            triggerChange();
                            scope.newTag = '';
                        }
                    });

                    if (alreadyExist) {
                        highlight(tag);
                    }
                }
            };

            ctrl.remove = function (tag) {
                scope.tags = _.without(scope.tags, tag);
                triggerChange();
            };

            ctrl.initAutocomplete = function (candidates) {
                var input = element.find('.tag-input');

                input.autocomplete({
                    source: candidates,
                    delay: 0,
                    autoFocus: true,
                    select: function (event, ui) {
                        ctrl.add(ui.item.value);
                    },
                    close: function () {
                        input.val(scope.newTag);
                    }
                });
            };

            function triggerChange() {
                if (scope.onChange) scope.onChange({'tags': scope.tags});
            }

            function highlight(value) {
                element.find(".tag-label:contains('" + value + "')").parents(".tag").effect('highlight');
            }
        }],
        link: function (scope, element, attributes, ctrl) {
            var hasAutocompletion = angular.isDefined(attributes.completionCandidates);
            scope.newTag = '';
            scope.remove = ctrl.remove;

            if (hasAutocompletion) {
                scope.$watch('completionCandidates', function (candidates) {
                    if (angular.isDefined(candidates)) {
                        ctrl.initAutocomplete(candidates);
                    }
                });
            }

            element.find('.tag-input').on('keypress', function (event) {
                // on enter
                if (event.which === 13) {
                    ctrl.add(scope.newTag);
                }
            });
        }
    }
});
