angular.module('xlrelease').factory('TimelineDisplay', ['Browser', 'CalendarConstants', 'CalendarDisplay', function (Browser, CalendarConstants, CalendarDisplay) {
    var timelineDayHeaderHeight = 26;
    var weekModeWidthThreshold = 60;
    var monthModeWidthThreshold = 10;

    function switchTimelineView(dayWidth, timelineSectionBody) {
        if (dayWidth < monthModeWidthThreshold) {
            timelineSectionBody.find('.day').removeClass('day-view').removeClass('week-view').addClass('month-view');
        } else if (dayWidth < weekModeWidthThreshold) {
            timelineSectionBody.find('.day').removeClass('day-view').removeClass('month-view').addClass('week-view');
        } else {
            timelineSectionBody.find('.day').removeClass('month-view').removeClass('week-view').addClass('day-view');
        }
    }

    function resizeTimelineWidth(timelineSectionBody) {
        var timelineSection = timelineSectionBody.parents('#timeline-section');
        var timelineSectionInnerWidth = timelineSection.width() - 2 * CalendarConstants.border;
        var dependenciesWidth = timelineSection.find('#incoming-dependencies:visible').width() + timelineSection.find('#outgoing-dependencies:visible').width();
        var timelineWidth = timelineSectionInnerWidth - dependenciesWidth;
        var numberOfDays = timelineSection.find('.day').length;
        var idealDayWidth = timelineWidth / numberOfDays - CalendarConstants.border;

        switchTimelineView(idealDayWidth, timelineSectionBody);
        computeDayWidth(timelineSectionBody, numberOfDays, _.parseInt(idealDayWidth), timelineWidth);

        timelineSectionBody.find('#timeline').css({'width': timelineWidth});
    }

    /**
     * Each browser manages decimal pixels differently.
     * So we set an integer day width and then we split the remaining pixels to the first days.
     */
    function computeDayWidth(element, numberOfDays, dayWidth, timelineWidth) {
        element.find('.day').width(dayWidth);

        var extraPixels = timelineWidth - numberOfDays * (dayWidth + CalendarConstants.border);
        for (var i = 0; i < extraPixels; i++) {
            element.find('.day:eq(' + i + ')').width(dayWidth + 1);
        }
    }

    function resizeTimelineHeight(timelineSectionBody, releasesDisplayed) {
        var height = timelineDayHeaderHeight + releasesDisplayed * CalendarConstants.releaseHeight;

        timelineSectionBody.css({'height': height});
        timelineSectionBody.find('#timeline').css({'height': height});
        timelineSectionBody.find('.day').css({'height': height});
    }

    function computeOffsetTop(element, index) {
        return element.parents('.section-body').prop('offsetTop') + timelineDayHeaderHeight + index * CalendarConstants.releaseHeight;
    }

    function computeHzlOffset(element, daysAfterFirst, dayRatio) {
        var offset;
        var days = element.parents('#timeline').find('.day');

        if (daysAfterFirst >= 0) {
            if (daysAfterFirst < days.length) {
                var day = days.eq(daysAfterFirst);
                offset = day.prop('offsetLeft') + day.outerWidth(true) * dayRatio;
            } else {
                var lastDay = days.eq(days.length - 1);
                offset = lastDay.prop('offsetLeft') + lastDay.outerWidth(true);
            }
        } else {
            offset = days.eq(0).prop('offsetLeft');
        }
        return offset;
    }

    return {
        resize: function (timelineSectionBody, releasesDisplayed) {
            // Height MUST be set BEFORE the width (fix IE8 bug ...)
            resizeTimelineHeight(timelineSectionBody, releasesDisplayed);
            resizeTimelineWidth(timelineSectionBody);
        },
        positionPlanItem: function (element, planItem, index, isPhase) {
            var borderSize = isPhase ? 0 : CalendarConstants.border;
            // Used with phases to make sure they don't spill over release borders:
            var phaseOffset = isPhase ? CalendarConstants.border : 0;

            var offsetLeft = computeHzlOffset(element, planItem.daysBetweenFirstAndStart, planItem.startRatio) + phaseOffset;
            var offsetRight = computeHzlOffset(element, planItem.daysBetweenFirstAndEnd, planItem.endRatio);

            if (offsetLeft !== offsetRight) {
                var width = offsetRight - offsetLeft - 2 * borderSize - phaseOffset;
                var offsetTop = computeOffsetTop(element, index) + phaseOffset;

                element.css({
                    top: offsetTop + 'px',
                    left: offsetLeft + 'px',
                    width: width + 'px'
                });

                if (!isPhase) {
                    CalendarDisplay.resizeReleaseTitle(element);
                }
            } else {
                element.hide();
            }
        },
        positionDependency: function (dependency, index) {
            dependency.css({
                top: computeOffsetTop(dependency, index) + 'px',
                left: dependency.parents('.dependencies').prop('offsetLeft') + 'px'
            });
        }
    };
}]);

angular.module('xlrelease').directive('resizeTimeline', ['TimelineDisplay', function (TimelineDisplay) {
    return function (scope, element) {
        scope.$watch('', function () {
            scope.$evalAsync(function () {
                TimelineDisplay.resize(element, scope.timeline.releases.length);
                scope.$broadcast('TIMELINE-RESIZED');
            });
        });

        $(window).resize(function () {
            scope.$apply(function () {
                TimelineDisplay.resize(element, scope.timeline.releases.length);
                scope.$broadcast('TIMELINE-RESIZED');
            });
        });

        scope.$on('RESIZE-TIMELINE', function () {
            TimelineDisplay.resize(element, scope.timeline.releases.length);
            scope.$broadcast('TIMELINE-RESIZED');
        });
    };
}]);

angular.module('xlrelease').directive('positionTimelineRelease', ['TimelineDisplay', function (TimelineDisplay) {
    return function (scope, element) {
        scope.$on('TIMELINE-RESIZED', function () {
            scope.$evalAsync(function () {
                TimelineDisplay.positionPlanItem(element, scope.release, scope.$index);
            })
        });
    }
}]);

angular.module('xlrelease').directive('positionTimelinePhase', ['TimelineDisplay', function (TimelineDisplay) {
    return function (scope, element) {
        scope.$on('TIMELINE-RESIZED', function () {
            scope.$evalAsync(function () {
                TimelineDisplay.positionPlanItem(element, scope.phase, scope.$parent.$index, true);
            })
        });
    }
}]);

angular.module('xlrelease').directive('positionOutgoingDependency', ['TimelineDisplay', function (TimelineDisplay) {
    return function (scope, element) {
        scope.$on('TIMELINE-RESIZED', function () {
            TimelineDisplay.positionDependency(element, scope.$index);
        });
    }
}]);

angular.module('xlrelease').directive('positionIncomingDependency', ['TimelineDisplay', function (TimelineDisplay) {
    return function (scope, element) {
        var incomingDependenciesOffset = scope.outgoingDependencies.length + 1;
        scope.$on('TIMELINE-RESIZED', function () {
            TimelineDisplay.positionDependency(element, scope.$index !== undefined ? (incomingDependenciesOffset + scope.$index) : -1);
        });
    }
}]);

angular.module('xlrelease').directive('positionDependencyHeader', ['TimelineDisplay', function (TimelineDisplay) {
    return function (scope, element) {
        scope.$on('TIMELINE-RESIZED', function () {
            TimelineDisplay.positionDependency(element, -1);
        });
    }
}]);

angular.module('xlrelease').directive('timelineResizer', [function () {
    return function (scope) {
        scope.resizeTimeline = function () {
            scope.$watch('', function () {
                scope.$evalAsync(function () {
                    scope.$emit('RESIZE-TIMELINE');
                })
            })
        }
    }
}]);
