'use strict';

angular.module('xlrelease').factory('Timeline', ['DateService', 'ReleasesService', function (DateService, ReleasesService) {
    function getDays(firstDay, lastDay) {
        var days = [];
        var numberOfDays = lastDay.diff(firstDay, 'days');
        for (var i = 0; i < numberOfDays; i++) {
            var currentDay = moment(firstDay).add('days', i);
            days.push({
                number: currentDay.date(),
                fullDate: currentDay,
                dateStatus: DateService.getDateStatus(currentDay)
            });
        }

        return days;
    }

    function getDaysAfterFirst(firstDay, day) {
        if (day.isAfter(firstDay)) {
            return day.diff(firstDay, 'days');
        }
        return -1;
    }

    function getReleases(firstDay, releases) {
        var timelineReleases = [];
        _.each(releases, function (release) {
            var startDate = moment(release.scheduledStartDate);
            var dueDate = moment(release.dueDate);

            var enhancedRelease = angular.copy(release);

            angular.extend(enhancedRelease, {
                startRatio: startDate.diff(DateService.atMidnight(startDate), 'days', true),
                endRatio: dueDate.diff(DateService.atMidnight(dueDate), 'days', true),
                daysBetweenFirstAndStart: getDaysAfterFirst(firstDay, startDate),
                daysBetweenFirstAndEnd: getDaysAfterFirst(firstDay, dueDate),
                displayablePhases: getDisplayablePhases(firstDay, enhancedRelease)
            });

            timelineReleases.push(enhancedRelease);
        });

        return timelineReleases;
    }

    function getDisplayablePhases(firstDay, release) {
        var enhancedPhases = [];

        _.each(release.phases, function (phase) {
            if (phase.scheduledStartDate && phase.dueDate) {
                var dates = ReleasesService.cropToReleaseDate(phase, release);
                var startDate = dates.startDate;
                var dueDate = dates.dueDate;

                angular.extend(phase, {
                    startRatio: startDate.diff(DateService.atMidnight(startDate), 'days', true),
                    endRatio: dueDate.diff(DateService.atMidnight(dueDate), 'days', true),
                    daysBetweenFirstAndStart: getDaysAfterFirst(firstDay, startDate),
                    daysBetweenFirstAndEnd: getDaysAfterFirst(firstDay, dueDate)
                });

                enhancedPhases.push(phase);
            }
        });

        return enhancedPhases;
    }

    return {
        init: function (currentRelease, releases) {
            if (currentRelease.scheduledStartDate && currentRelease.dueDate) {
                var firstDay = DateService.atMidnight(currentRelease.scheduledStartDate);
                var lastDay = DateService.atMidnight(moment(currentRelease.dueDate).add('days', 1));

                return {
                    days: getDays(firstDay, lastDay),
                    releases: getReleases(firstDay, releases)
                }
            }

            return { days: [], releases: [] }
        }
    };
}]);
