'use strict';

angular.module('xlrelease').directive('worksheetScrolling', function (worksheetService) {

    var INCLUDE_MARGIN = true;
    var releaseHeaderHeight, navigatorTogglerWidth, wrapper, worksheetWidth, phaseMarginLeft, phaseMarginRight, phaseWidth, scope;

    function updateOverflowFlag() {
        var contentWidth = wrapper.parents(".view").width();
        var wrapperWidth = wrapper.width();

        scope.hasOverflow = contentWidth < wrapperWidth;
        if (!scope.hasOverflow) {
            scope.showNavigator = false;
        }
    }

    function highlightCurrentPhasesLinks() {
        var phaseLinksConfig = worksheetService.computeWhichPhasesAreHighlighted({
            phaseWidth: phaseWidth,
            scrollLeft: wrapper.parents(".view").scrollLeft(),
            worksheetWidth: worksheetWidth,
            phaseMarginLeft: phaseMarginLeft,
            phaseMarginRight: phaseMarginRight
        });

        highlightPhasesLinks(phaseLinksConfig.firstPhaseVisibleIndex, phaseLinksConfig.numberOfDisplayedPhase);
    }

    function highlightPhasesLinks(firstPhaseVisibleIndex, numberOfDisplayedPhase) {
        var navigator = wrapper.siblings("#navigator");

        navigator.find(".phase-link-container").removeClass('highlight');
        for (var i = firstPhaseVisibleIndex; i < firstPhaseVisibleIndex + numberOfDisplayedPhase; i++) {
            navigator.find(".phase-link-container:nth-child(" + (i + 1) + ")").addClass("highlight");
        }
    }

    function computeDomElementsDimensions() {
        var phase = wrapper.find(".phase");
        phaseMarginLeft = _.parseInt(phase.css("margin-left"));
        phaseMarginRight = _.parseInt(phase.css("margin-right"));
        phaseWidth = phase.outerWidth(INCLUDE_MARGIN);
        navigatorTogglerWidth = _.parseInt(wrapper.siblings("#navigator-toggler").width());
        releaseHeaderHeight = _.parseInt(wrapper.find("#release-header").outerHeight());
    }

    function computeWorksheetWidth() {
        worksheetWidth = wrapper.parents(".view").width() - wrapper.siblings("#navigator").width();

        var phases = wrapper.find(".phase");
        var wrapperWidth = 0;
        for (var i = 0; i < phases.length - 1; i++) {
            wrapperWidth += $(phases[i]).outerWidth(INCLUDE_MARGIN);
        }
        wrapperWidth += wrapper.parents(".view").outerWidth(INCLUDE_MARGIN) - navigatorTogglerWidth;
        wrapper.css({'min-width': wrapperWidth + 'px'});
    }

    function scrollingCtrl() {
        var ctrl = this;
        ctrl.setupScrollToPhase = function () {
            wrapper.siblings("#navigator").find(".phase-link").each(function (phaseIndex, item) {
                $(item).unbind('click');
                $(item).on('click', function () {
                    var phaseLink = $(this);
                    ctrl.scrollTo(wrapper.find(".phase").eq(phaseIndex), null, function () {
                        phaseLink.addClass("scrolled");
                    })
                });
            });
        };
        ctrl.scrollTo = function (phase, task, callback) {
            var contentMarginLeft = _.parseInt(wrapper.find("#release-content").css("margin-left"));
            contentMarginLeft = isNaN(contentMarginLeft) ? 0 : contentMarginLeft;

            var scrollTop = 0;
            if (task != null) {
                var bottomScrollbarHeight = 18;
                var container = task.parents('.task-container');

                var isTaskVisible = (container.prop('offsetTop') + container.height()) <= (wrapper.parents(".view").height() - bottomScrollbarHeight);
                scrollTop = !isTaskVisible ? container.prop('offsetTop') - releaseHeaderHeight : scrollTop;
            }

            var scrollParams = {
                scrollLeft: phase.prop('offsetLeft') - (navigatorTogglerWidth + contentMarginLeft + phaseMarginLeft),
                scrollTop: scrollTop
            };

            worksheetService.scroll(wrapper.parents(".view"), scrollParams, callback);
        }
    }

    return {
        controller: scrollingCtrl,
        link: function (currentScope, element, attr, ctrl) {
            wrapper = element;
            scope = currentScope;

            $(window).resize(function () {
                scope.$apply(function () {
                    computeWorksheetWidth();
                    updateOverflowFlag();
                    highlightCurrentPhasesLinks();
                });
            });

            scope.$watch('release.phases', function (newPhases, oldPhases) {
                if (angular.isDefined(newPhases)) {
                    computeDomElementsDimensions();
                    computeWorksheetWidth();
                    updateOverflowFlag();
                    highlightCurrentPhasesLinks();

                    ctrl.setupScrollToPhase();
                    wrapper.parents(".view").unbind('scroll');
                    wrapper.parents(".view").scroll(function () {
                        highlightCurrentPhasesLinks();
                    });

                    if (angular.isDefined(oldPhases)) {
                        if (worksheetService.isAddedPhaseLast(newPhases, oldPhases)) {
                            ctrl.scrollTo(wrapper.find(".phase").eq(newPhases.length - 1), null);
                        }
                    }
                }
            }, true);

            scope.$watch('showNavigator', function (isNavigatorOpen, wasNavigatorOpen) {
                if (isNavigatorOpen === wasNavigatorOpen) {
                    return;
                }
                computeWorksheetWidth();
                highlightCurrentPhasesLinks();
            });
        }
    };
});

angular.module('xlrelease').directive('scrollToMe', function () {
    return {
        require: '^worksheetScrolling',
        link: function (scope, task, attr, scrollingCtrl) {
            if (!scope.init.hasScrolledToCurrentTask) {
                // DOM must be computed before scrolling
                scope.$watch('', function () {
                    scope.$evalAsync(function () {
                        scope.init.hasScrolledToCurrentTask = true;
                        scrollingCtrl.scrollTo(task.parents(".phase"), task);
                    })
                });
            }
        }
    }
});
