'use strict';

angular.module('xlrelease').controller('taskDetailsController', ['$scope', 'Backend', 'serverTitleFilter', 'TasksService', '$location', 'Authenticator', 'FlagService',
    function ($scope, Backend, serverTitleFilter, TasksService, $location, Authenticator, FlagService) {

    var refreshTasksOnModalExit = false;

    $scope.data = {
        newComment: '',
        showCommentForm: false
    };

    $scope.saveTask = function (task) {
        var updatedTask = {
            id: task.id,
            title: task.title,
            description: task.description,
            dueDate: task.dueDate,
            scheduledStartDate: task.scheduledStartDate,
            team: task.team,
            owner: task.owner,
            deploymentPackage: task.deploymentPackage,
            environment: task.environment,
            server: task.server,
            username: task.username,
            password: task.password,
            taskType: task.taskType,
            addresses: task.addresses,
            subject: task.subject,
            body: task.body,
            script: task.script,
            flag: task.flag,
            scriptDefinitionType: task.scriptDefinitionType,
            inputProperties: task.inputProperties,
            outputProperties: task.outputProperties
        };
        Backend.put('tasks/' + updatedTask.id, updatedTask).success(function (savedTask) {
            task.password = savedTask.password;
            task.outputProperties = savedTask.outputProperties;
        });
    };

    $scope.editTask = function (task, release) {
        loadUserNames(release);
        loadTeams(task.id, release);
        loadComments(task.id);
        if (TasksService.isDeployitTask(task) && !TasksService.isTaskInProgress(task) && !TasksService.isTaskReadOnly(task)) {
            loadDeployitData(task.server);
        }
        loadCustomType(task);
    };

    function loadUserNames (release) {
        $scope.userNames = [];
        if (Authenticator.hasEditTaskPermission(release) && Authenticator.hasReassignTaskPermission(release)) {
            Backend.get('users').success(function (userNames) {
                $scope.userNames = userNames;
            });
        }
    }

    function loadTeams(taskId, release) {
        $scope.teams = [];
        if (Authenticator.hasEditTaskPermission(release) && Authenticator.hasReassignTaskPermission(release)) {
            Backend.get('tasks/' + taskId + '/teams/assignable').success(function (teams) {
                $scope.teams = _.map(teams, function (team) {
                    return {id: team.teamName, title: team.teamName};
                });
            });
        }
    }

    function loadComments(taskId) {
        $scope.comments = [];
        Backend.get('tasks/' + taskId + '/comments').success(function (comments) {
            $scope.comments = comments;
        });
    }

    function loadDeployitData (deployitServerId) {
        loadDeployitServers();
        $scope.loadDeployitAutocompleteList(deployitServerId);
    }

    $scope.loadDeployitAutocompleteList = function (deployitServerId) {
        if (deployitServerId) {
            Backend.get('deployit/servers/' + deployitServerId + '/packages').success(function (packages) {
                $scope.deployitPackages =  packages;
            });
            Backend.get('deployit/servers/' + deployitServerId + '/environments').success(function (environments) {
                $scope.deployitEnvironments = environments;
            });
        }
    };

    function loadDeployitServers() {
        Backend.get('deployit/servers/ids').success(function (serversIds) {
            $scope.deployitServers = _.map(serversIds, function (serverId) {
                return {id: serverId, title: serverTitleFilter(serverId)};
            });
        });
    }

    function loadCustomType(task) {
        if (task.scriptDefinitionType) {
            Backend.get('tasks/python-script-definitions/' + task.scriptDefinitionType).success(function (definition) {
                $scope.scriptDefinition = definition;
            });
        }
    }

    $scope.saveComment = function (comment) {
        Backend.put('comments/' + comment.id, comment).success(function (updatedComment) {
            comment.date = updatedComment.date
        });
    };

    $scope.addComment = function (task, commentText) {
        var newComment = {
            text: commentText
        };
        Backend.post('tasks/' + task.id + '/comments', newComment).success(function (comment) {
            $scope.data.newComment = '';
            $scope.comments.push(comment);
            task.numberOfComments++;
            $scope.data.showCommentForm = false;
        });
    };

    $scope.flag = function (flagStatus) {
        $scope.task.flag.status = flagStatus;
        if (!FlagService.isTaskFlagged($scope.task)) {
            $scope.task.flag.comment = '';
        }
        $scope.saveTask($scope.task);
    };

    $scope.removeOwner = function () {
        $scope.task.owner = null;
        $scope.saveTask($scope.task);
        refreshTasksOnModalExit = true;
    };

    $scope.updateOwner = function (task) {
        $scope.saveTask(task);
        refreshTasksOnModalExit = true;
    };

    $scope.removeTeam = function () {
        $scope.task.team = null;
        $scope.saveTask($scope.task);
        refreshTasksOnModalExit = true;
    };

    $scope.updateTeam = function (task) {
        $scope.saveTask(task);
        refreshTasksOnModalExit = true;
    };

    $scope.refreshTasks = function () {
        if (refreshTasksOnModalExit) {
            $scope.loadTasks();
            refreshTasksOnModalExit = false;
        }
    };

    $scope.viewTaskInRelease = function (releaseId) {
        $location.path('releases/' + releaseId);
    };

    $scope.canWorkOnTask = function (release) {
        if (TasksService.isTaskReadOnly($scope.task)) {
            return false;
        }
        return Authenticator.hasEditTaskPermission(release) || Authenticator.isOwner($scope.task.owner);
    };

    $scope.canEditTask = function (release) {
        return Authenticator.hasEditTaskPermission(release) && !TasksService.isTaskReadOnly($scope.task);
    };

    $scope.canClaimTask = function (task, release) {
        return !$scope.belongsToMe(task) && $scope.canEditTask(release) && Authenticator.hasReassignTaskPermission(release);
    };

    // workaround for scope issues with ng-if
    $scope.switchStartTaskMode = function () {
        $scope.startTaskMode = !$scope.startTaskMode;
    };

    $scope.canTaskBeScheduled = function (task) {
        return TasksService.isTaskPlanned(task) || TasksService.isTaskPending(task);
    };

    $scope.assignToMe = function (task) {
        task.owner = Authenticator.getUsername();
        $scope.updateOwner(task);
    };

    $scope.belongsToMe = function (task) {
        return TasksService.isAssignedToUser(task, Authenticator.getUsername());
    };

    $scope.isStatusUpdatable = TasksService.isStatusUpdatable;
    $scope.isTaskInProgress = TasksService.isTaskInProgress;
    $scope.isGateTask = TasksService.isGateTask;
    $scope.isDeployitTask = TasksService.isDeployitTask;
    $scope.isNotificationTask = TasksService.isNotificationTask;
    $scope.isScriptTask = TasksService.isScriptTask;
    $scope.isParallelGroup = TasksService.isParallelGroup;
    $scope.isGateClosed = TasksService.isGateClosed;
    $scope.isTaskReadOnly = TasksService.isTaskReadOnly;
    $scope.isTaskFailed = TasksService.isTaskFailed;
    $scope.isTaskPending = TasksService.isTaskPending;
    $scope.isTaskAborted = TasksService.isTaskAborted;
    $scope.isTaskCompleted = TasksService.isTaskCompleted;
    $scope.isTaskSkipped = TasksService.isTaskSkipped;
    $scope.hasOwner = TasksService.hasOwner;
}]);
