'use strict';

angular.module('xlrelease').factory('TasksService', ['Backend', 'DateService', function (Backend, DateService) {
    function isTaskPlanned(task) {
        return 'PLANNED' === task.status;
    }

    function isTaskPending(task) {
        return 'PENDING' === task.status;
    }

    function isTaskInProgress(task) {
        return 'IN_PROGRESS' === task.status;
    }

    function isTaskCompleted(task) {
        return 'COMPLETED' === task.status || 'ABORTED' === task.status;
    }

    function isTaskSkipped(task) {
        return 'SKIPPED' === task.status;
    }

    function isDeployitTask(task) {
        return 'xlrelease.DeployitTask' === task.taskType;
    }

    function isGateTask(task) {
        return 'xlrelease.GateTask' === task.taskType;
    }

    function isNotificationTask(task) {
        return 'xlrelease.NotificationTask' === task.taskType;
    }

    function isScriptTask(task) {
        return 'xlrelease.ScriptTask' === task.taskType;
    }

    function isParallelGroup(task) {
        return 'xlrelease.ParallelGroup' === task.taskType;
    }

    function isManualTask(task) {
        return 'xlrelease.Task' === task.taskType;
    }

    function isParallelGroupInProgress(task) {
        return isParallelGroup(task) && isTaskInProgress(task);
    }

    function someGateConditionUnchecked(task) {
        return _.some(task.conditions, function (condition) {
            return condition.checked === false
        });
    }

    function someGateDependencyUnresolved(task) {
        return _.some(task.dependencies, function (dependency) {
            return dependency.resolved === false
        });
    }

    function isGateClosed(task) {
        return isGateTask(task) && (someGateConditionUnchecked(task) || someGateDependencyUnresolved(task));
    }

    function hasOwner(task) {
        return angular.isDefined(task.owner) && task.owner !== null;
    }

    function hasTeam(task) {
        return angular.isDefined(task.team) && task.team !== null;
    }

    function updateTaskStatusTo(status, id, commentText) {
        var comment = {
            text: commentText
        };
        return Backend.post('tasks/' + id + '/' + status, comment);
    }

    function isAutomated(task) {
        return task.automated;
    }

    return {
        hasOwner: hasOwner,
        hasTeam: hasTeam,
        isTaskPlanned: isTaskPlanned,
        isTaskPending: isTaskPending,
        isTaskInProgress: isTaskInProgress,
        isTaskCompleted: isTaskCompleted,
        isTaskSkipped: isTaskSkipped,
        isDeployitTask: isDeployitTask,
        isAutomated: isAutomated,
        isGateTask: isGateTask,
        isNotificationTask: isNotificationTask,
        isScriptTask: isScriptTask,
        isManualTask: isManualTask,
        isParallelGroup: isParallelGroup,
        isParallelGroupInProgress: isParallelGroupInProgress,
        isGateClosed: isGateClosed,
        isTaskReadOnly: function (task) {
            return isTaskCompleted(task) || isTaskSkipped(task);
        },
        isTaskFailed: function (task) {
            return 'FAILED' === task.status;
        },
        isTaskAborted: function (task) {
            return 'ABORTED' === task.status;
        },
        isStatusUpdatable: function (task) {
            return isTaskInProgress(task) && hasOwner(task) && !isAutomated(task);
        },
        completeTask: function (taskId, commentText) {
            return updateTaskStatusTo('complete', taskId, commentText);
        },
        failTask: function (taskId, commentText) {
            return updateTaskStatusTo('fail', taskId, commentText);
        },
        skipTask: function (taskId, commentText) {
            return updateTaskStatusTo('skip', taskId, commentText);
        },
        retryTask: function (taskId, commentText) {
            return updateTaskStatusTo('retry', taskId, commentText);
        },
        startNow: function (taskId, commentText) {
            return updateTaskStatusTo('startNow', taskId, commentText);
        },
        isTaskDelayed: function (task) {
            if (!isTaskPlanned(task)) return false;
            return DateService.isOverdue(task.scheduledStartDate) || DateService.isOverdue(task.dueDate);
        },
        isAssignedToUser: function (task, user) {
            return hasOwner(task) && task.owner === user;
        },
        isAssignedToTeam: function (task, team) {
            return hasTeam(task) && task.team === team;
        }
    };
}]);

