#
# Copyright (c) 2021. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of Digital.ai, Inc. and its affiliates, subsidiaries, and licensors.
#
import json
from xlrelease.HttpRequest import HttpRequest

class RestClient:

    def __init__(self, argo_server=None):
        self.httpRequest  = HttpRequest(argo_server, verify=False)
        self.username = argo_server['username']
        self.password = argo_server['password']
        self.auth_token = argo_server['authToken']

    def getBearerToken(self):
        if self.auth_token is not None and self.auth_token != "":
            print "The provided authToken can be used as a bearer token"
            return self.auth_token
        # Create POST body
        content = {
            'username': self.username,
            'password': self.password
        }
        response = self.httpRequest.post('/api/v1/session', json.dumps(content),
                                contentType='application/json', headers=None)
        if response.isSuccessful():
            response_json = json.loads(response.getResponse())
            bearer_token = response_json['token']
        else:
            raise ValueError('Failed to get bearer token %s:%s' % (response.getStatus(),response.getResponse()))
        return bearer_token

    def checkConnection(self):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        response = self.httpRequest.get('/api/v1/projects', contentType='application/json', headers=headers)

        if not response.isSuccessful():
            raise ValueError('Failed to connect the server %s:%s' % (response.getStatus(), response.getResponse()))

    def listProjects(self):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        response = self.httpRequest.get('/api/v1/projects', contentType='application/json', headers=headers)
        result = {}
        projects = []
        if response.isSuccessful():
            responseObj = json.loads(response.getResponse())
            items = responseObj["items"]
            for item in items:
                project = {}
                spec = item["spec"]
                metadata = item["metadata"]
                project["name"] = metadata.get("name")
                project["namespace"] = metadata.get("namespace")
                project["description"] = spec.get("description")
                project["sourceRepos"] = spec.get("sourceRepos")
                project["destinations"] = spec.get("destinations")
                projects.append(project)
        else:
            raise ValueError('Failed to get project list %s:%s' % (response.getStatus(),response.getResponse()))
        result["projects"] = projects
        return result

    def listRepositories(self):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        response = self.httpRequest.get('/api/v1/repositories', contentType='application/json', headers=headers)
        repos = None
        if response.isSuccessful():
            repositories = json.loads(response.getResponse())
            repos = []
            items = repositories["items"]
            for item in items:
                repo = item.get("repo")
                if repo is not None :
                    repos.append(repo)
        else:
            raise ValueError('Failed to get repositories list %s:%s' % (response.getStatus(),response.getResponse()))
        return repos

    def addRepository(self, url, username, password, sshKey):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        if sshKey :
            body = {
                "url": url,
                "sshPrivateKey": sshKey
            }
        else:
            body = {
                "type": "git",
                "repo": url,
                "username": username,
                "password": password
            }
        response = self.httpRequest.post('/api/v1/repositories', json.dumps(body), contentType='application/json', headers=headers)

        if not response.isSuccessful():
           raise ValueError('Failed to add repository list %s:%s' % (response.getStatus(),response.getResponse()))

    def createApplication(self, appName, project, url, path, revision, server, namespace,
                          autoSync, pruneResources, selfHeal, skipValidation, pruneLast,
                          autoCreateNamespace, applyOutOfSyncOnly, prunePropagationPolicy, replace):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        body = {
            "metadata": {
                "name": appName
            },
            "spec": {
                "destination": {
                    "namespace": namespace,
                    "server": server
                },
                "project": project,
                "source": {
                    "path": path,
                    "repoURL": url,
                    "targetRevision": revision
                }
            }
        }
        syncPolicy = {}
        if autoSync:
            automated = {}
            automated["prune"] = pruneResources
            automated["selfHeal"] = selfHeal
            syncPolicy["automated"] = automated

        syncOptions = []
        syncOptions.append("Validate=%s" % (not skipValidation))
        syncOptions.append("PruneLast=%s" % (pruneLast))
        syncOptions.append("CreateNamespace=%s" % (autoCreateNamespace))
        syncOptions.append("ApplyOutOfSyncOnly=%s" % (applyOutOfSyncOnly))
        syncOptions.append("Replace=%s" % (replace))
        syncOptions.append("PrunePropagationPolicy=%s" % (prunePropagationPolicy))
        syncPolicy["syncOptions"] = syncOptions

        specObj = body["spec"]
        specObj["syncPolicy"] = syncPolicy
        body["spec"] = specObj

        response = self.httpRequest.post('/api/v1/applications', json.dumps(body), contentType='application/json', headers=headers)

        if not response.isSuccessful():
            raise ValueError('Failed to create app %s:%s' % (response.getStatus(),response.getResponse()))

        return response.getResponse()

    def syncApplication(self, appName):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        endpoint = "/api/v1/applications/" + appName + "/sync"
        response = self.httpRequest.post(endpoint, '', contentType='application/json', headers=headers)

        if not response.isSuccessful():
            raise ValueError('Failed to sync app list %s:%s' % (response.getStatus(),response.getResponse()))
        else:
            responseObj = json.loads(response.getResponse())
            statusObj = responseObj["status"]
            syncObj = statusObj["sync"]
            status = syncObj["status"]

        return status

    def getSyncStatus(self, appName):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        endpoint = "/api/v1/applications/" + appName
        response = self.httpRequest.get(endpoint, contentType='application/json', headers=headers)
        if not response.isSuccessful():
            raise ValueError('Failed to get sync status %s:%s' % (response.getStatus(),response.getResponse()))

        deployment_status = {}

        responseObj = json.loads(response.getResponse())
        statusObj = responseObj["status"]
        sync = statusObj["sync"]
        syncStatus = sync["status"]
        deployment_status["syncStatus"] = syncStatus
        healthObj = statusObj["health"]
        healthStatus = healthObj["status"]

        deployment_status["syncStatus"] = syncStatus
        deployment_status["healthStatus"] = healthStatus

        return deployment_status

    def createProject(self, projectName, sourceRepos, destinations):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        body = { "upsert": True }
        project = {
            "metadata": {
                "name": projectName
            }
        }
        spec = {}
        if sourceRepos:
            sourceRepoList = []
            for repo in sourceRepos:
                sourceRepoList.append(repo)
            spec["sourceRepos"] = sourceRepoList

        if destinations:
            destinationList = []
            for serverKey in destinations:
                destObj =  {
                    "namespace": destinations[serverKey],
                    "server": serverKey
                }
                destinationList.append(destObj)
            spec["destinations"] = destinationList

        project["spec"] = spec
        body["project"] = project

        response = self.httpRequest.post('/api/v1/projects', json.dumps(body), contentType='application/json', headers=headers)

        if not response.isSuccessful():
            raise ValueError('Failed to create project %s:%s' % (response.getStatus(),response.getResponse()))

        return response.getResponse()

    def getAppDetails(self, appName, bearer_token):
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        endpoint = "/api/v1/applications/" + appName
        response = self.httpRequest.get(endpoint, contentType='application/json', headers=headers)

        if not response.isSuccessful():
            raise ValueError('Failed to get application details %s:%s' % (response.getStatus(),response.getResponse()))

        return json.loads(response.getResponse())

    def getResourceDetails(self, appName, resourceName, bearer_token):
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        endpoint = "/api/v1/applications/" + appName + "/managed-resources?name="+resourceName
        response = self.httpRequest.get(endpoint, contentType='application/json', headers=headers)

        if not response.isSuccessful():
            raise ValueError('Failed to get resource details %s:%s' % (response.getStatus(),response.getResponse()))

        return json.loads(response.getResponse())

    def getResourceTree(self, appName):
        bearer_token = self.getBearerToken()
        headers = {'Authorization': 'Bearer %s' % bearer_token}
        endpoint = "/api/v1/applications/" + appName+"/resource-tree"
        response = self.httpRequest.get(endpoint, contentType='application/json', headers=headers)

        if not response.isSuccessful():
            raise ValueError('Failed to get sync status %s:%s' % (response.getStatus(), response.getResponse()))

        return json.loads(response.getResponse())


