package com.xebialabs.xlrelease.script

import java.util.{Map => JMap}

import com.xebialabs.platform.script.jython.JythonException
import com.xebialabs.xlplatform.endpoints.actors.ScriptDone
import com.xebialabs.xlrelease.domain.{Task, CustomScriptTask, ReleaseTrigger, ScriptTask}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

/**
 * This service is used for testing of script execution.
 */
@Service
class ScriptTestService @Autowired() (synchronousScriptService: SynchronousScriptService,
                                      endpointExecutor: EndpointJythonScriptTestExecutor) {

  /**
   * Executes given script task. The task and related phase and release must be persisted into repository
   * if you use methods such as <code>getCurrentTask()</code>.
   * @return script output if the script execution succeeded
   */
  @throws[JythonException]("JythonException if script execution failed")
  def executeScriptTask(task: ScriptTask): String = {
    synchronousScriptService.executeScriptTask(task)
  }

  /**
   * Executes given custom script task. The task content is loaded as normally in XL Release from the classpath
   * depending on the custom script task type.
   * @param task    the custom script task of an existing type.
   * @return      output of task execution. The output properties are also written into provided task, e.g.
   *              <code>customScriptTask.getPythonScript().getProperty("result")</code>.
   */
  @throws[JythonException]("JythonException if script execution failed")
  def executeCustomScriptTask(task: CustomScriptTask): String = {
    synchronousScriptService.executeCustomScriptTask(task)
  }

  /**
   * Executes a precondition on given task and returns the result.
   * @param task  the task with precondition.
   * @return      boolean result of precondition execution.
   */
  @throws[JythonException]("JythonException if script execution failed")
  def executePrecondition(task: Task): Boolean = {
    synchronousScriptService.executePrecondition(task)
  }

  /**
   * Executes given release trigger. You can use <code>trigger.getProperty("myProperty")</code> to verify updated
   * property values after execution. Note that no release is started as a result of this call.
   * @param trigger the release trigger to execute
   */
  @throws[JythonException]("JythonException if script execution failed")
  def executeTrigger(trigger: ReleaseTrigger): Unit = {
    synchronousScriptService.executeTrigger(trigger)
  }

  /**
   * Executes a given endpoint Jython script and returns the result.
   * @param scriptLocation Jython script location in classpath.
   * @param query query parameters used when invoking the HTTP endpoint.
   * @param jsonEntity JSON entity submitted to the HTTP endpoint, or <code>null</code> if empty.
   * @param user user logged in when invoking HTTP endpoint.
   * @return the script execution result as it would be if a real endpoint would be executed.
   */
  @throws[JythonException]("JythonException if script execution failed")
  def executeEndpointScript(scriptLocation: String, query: JMap[String, AnyRef], jsonEntity: String, user: String): ScriptDone = {
    endpointExecutor.executeEndpointScript(scriptLocation, query, jsonEntity, user)
  }

}
