package com.xebialabs.xlrelease.script;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.Collection;
import javax.script.ScriptContext;
import javax.script.ScriptException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.google.common.annotations.VisibleForTesting;

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.platform.script.jython.JythonException;
import com.xebialabs.platform.script.jython.ThreadLocalWriterDecorator;
import com.xebialabs.xlrelease.domain.CustomScriptTask;
import com.xebialabs.xlrelease.domain.ReleaseTrigger;
import com.xebialabs.xlrelease.domain.ScriptTask;
import com.xebialabs.xlrelease.domain.Task;

import static com.google.common.base.Throwables.propagate;
import static java.lang.String.format;

@Service
public class SynchronousScriptService {

    private ScriptService scriptService;
    private ScriptLifeCycle scriptLifeCycle;

    @Autowired
    public SynchronousScriptService(ScriptService scriptService, ScriptLifeCycle scriptLifeCycle) {
        this.scriptService = scriptService;
        this.scriptLifeCycle = scriptLifeCycle;
    }

    public SynchronousScriptService() {}

    public String executeScriptTask(ScriptTask task) throws JythonException {
        ScriptContext scriptContext = scriptService.prepareScriptTaskContext(task, scriptService.getExecutionLog());
        return executeScriptSynchronously(task.getScript(), scriptContext, task, true);
    }

    public String executeCustomScriptTask(CustomScriptTask task) throws JythonException {
        try {
            String script = task.getPythonScript().getScript();
            return executeCustomScriptTask(task, script);
        } catch (IOException e) {
            throw new JythonException(format("Error while loading script on task '%s'", task.getId()), e);
        }
    }

    @VisibleForTesting
    String executeCustomScriptTask(CustomScriptTask task, String scriptContent) throws JythonException {
        ScriptContext scriptContext = scriptService.prepareCustomScriptTaskContext(task, scriptService.getExecutionLog());
        String output = executeScriptSynchronously(scriptContent, scriptContext, task, false);
        saveOutputPropertiesInMemory(task, scriptContext);
        return output;
    }

    public boolean executePrecondition(Task task) throws JythonException {
        scriptLifeCycle.register(task.getExecutionId());

        StringWriter stringWriter = new StringWriter();
        scriptService.getExecutionLog().registerWriter(new PrintWriter(stringWriter));

        String script = task.getPrecondition();

        try {
            ScriptContext scriptContext = scriptService.preparePreconditionContext(task, scriptService.getExecutionLog());

            Object statementResult = scriptService.executeScript(script, scriptContext, task, true);

            Object resultVariable = scriptContext.getAttribute("result");

            if (statementResult == null && resultVariable == null) {
                throw new JythonException("Precondition did not return anything", null);
            }
            return isTrue(statementResult) || isTrue(resultVariable);
        } catch (Exception exception) {
            closeQuietly(scriptService.getExecutionLog());
            throw new JythonException("Error while running script, the output so far:\n" + stringWriter.toString(), exception);
        } finally {
            closeQuietly(scriptService.getExecutionLog());
            scriptLifeCycle.unregister(task.getExecutionId());
        }

    }

    public String executeTrigger(ReleaseTrigger releaseTrigger) throws JythonException {
        StringWriter stringWriter = new StringWriter();
        scriptService.getExecutionLog().registerWriter(new PrintWriter(stringWriter));

        try {
            ScriptContext context = scriptService.makeScriptContext(releaseTrigger);
            scriptService.executeScript(releaseTrigger.getScript(), context);
            scriptService.setPropertiesFromScriptContext(context, releaseTrigger);
        } catch (Exception e) {
            throw new JythonException("Error during trigger execution, the output so far:\n" + stringWriter.toString(), e);
        } finally {
            closeQuietly(scriptService.getExecutionLog());
        }

        return stringWriter.toString();
    }

    private String executeScriptSynchronously(String script, ScriptContext scriptContext, Task task, boolean checkPolicyPermissions)
            throws JythonException {
        scriptLifeCycle.register(task.getExecutionId());

        StringWriter stringWriter = new StringWriter();
        scriptService.getExecutionLog().registerWriter(new PrintWriter(stringWriter));

        try {
            scriptService.executeScript(script, scriptContext, task, checkPolicyPermissions);
        } catch (ScriptException exception) {
            if (!scriptService.isSystemExit0(exception)) {
                closeQuietly(scriptService.getExecutionLog());
                throw new JythonException("Error while running script, the output so far:\n" + stringWriter.toString(), exception);
            }
        } catch (Exception exception) {
            closeQuietly(scriptService.getExecutionLog());
            throw new JythonException("Error while running script, the output so far:\n" + stringWriter.toString(), exception);
        } finally {
            closeQuietly(scriptService.getExecutionLog());
            scriptLifeCycle.unregister(task.getExecutionId());
        }

        return stringWriter.toString();
    }

    private void saveOutputPropertiesInMemory(CustomScriptTask task, ScriptContext scriptContext) {
        Collection<PropertyDescriptor> outputProperties = task.getPythonScript().getOutputProperties();
        for (PropertyDescriptor propertyDescriptor : outputProperties) {
            String propertyName = propertyDescriptor.getName();
            Object attr = scriptContext.getAttribute(propertyName);
            if (attr != null) {
                task.getPythonScript().setProperty(propertyName, attr.toString());
            }
        }
    }

    private void closeQuietly(ThreadLocalWriterDecorator log) {
        try {
            log.getWriter().close();
        } catch (IOException exception) {
            propagate(exception);
        }
    }

    private boolean isTrue(Object preconditionResult) {
        return preconditionResult instanceof Boolean && (Boolean) preconditionResult;
    }

}
