package com.xebialabs.xlrelease.rules;

import java.util.List;
import org.junit.rules.ExternalResource;
import org.junit.rules.TestRule;
import org.junit.runner.Description;
import org.junit.runners.model.Statement;
import org.springframework.security.authentication.TestingAuthenticationToken;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.core.context.SecurityContextImpl;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;

import com.xebialabs.deployit.security.PermissionEditor;
import com.xebialabs.deployit.security.Permissions;
import com.xebialabs.deployit.security.Role;
import com.xebialabs.deployit.security.RoleService;
import com.xebialabs.deployit.security.permission.Permission;

import static com.xebialabs.deployit.jcr.JcrConstants.GLOBAL_SECURITY_ALIAS;
import static com.xebialabs.deployit.security.PermissionEnforcer.ROLE_ADMIN;
import static com.xebialabs.deployit.security.permission.PlatformPermissions.ADMIN;
import static org.assertj.core.util.Lists.newArrayList;

public class LoginRule extends ExternalResource {

    private static final String DEFAULT_PASSWORD = "password";

    private final String username;

    private final String password;

    private List<String> authorities;

    private LoginRule(String username, String password, String... authorities) {
        this.username = username;
        this.password = password;
        this.authorities = newArrayList(authorities);
    }

    public static LoginRule loginWith(String username) {
        return new LoginRule(username, DEFAULT_PASSWORD);
    }

    public static LoginRule loginWithRoleAdmin(String username) {
        return new LoginRule(username, DEFAULT_PASSWORD, ROLE_ADMIN);
    }

    public void logout() {
        Permissions.clearSecurityContext();
    }

    @Override
    public void before() throws Throwable {
        SecurityContextImpl context = new SecurityContextImpl();
        context.setAuthentication(new TestingAuthenticationToken(username, password, authorities.toArray(new String[authorities.size()])));
        SecurityContextHolder.setContext(context);
    }

    @Override
    public void after() {
        Permissions.clearSecurityContext();
    }

    public static void grantAdminPermissionTo(String username, PermissionEditor permissionEditor, RoleService roleService) {
        Role roleWithAdminPermission = new Role(roleService.getRoles().size(), "RoleWithAdminPermission");
        roleWithAdminPermission.getPrincipalsAssigned().add(username);
        Multimap<Role, Permission> rolePermission = ArrayListMultimap.create();
        rolePermission.put(roleWithAdminPermission, ADMIN);
        permissionEditor.editPermissions(GLOBAL_SECURITY_ALIAS, rolePermission);
        roleService.writeRoleAssignments(Lists.newArrayList(roleWithAdminPermission));
    }

    public static void addUserToRole(String username, String roleName, RoleService roleService) {
        Role role = new Role(roleService.getRoles().size(), roleName);
        role.getPrincipalsAssigned().add(username);
        roleService.writeRoleAssignments(Lists.newArrayList(role));
    }

    public static void clearPermissions(PermissionEditor permissionEditor, RoleService roleService) {
        Multimap<Role, Permission> rolePermission = ArrayListMultimap.create();
        permissionEditor.editPermissions(GLOBAL_SECURITY_ALIAS, rolePermission);
        roleService.writeRoleAssignments(Lists.<Role>newArrayList());
    }
}
