import React, { MouseEvent, useState } from 'react';
import ReactMarkdown from 'react-markdown';
import classNames from 'classnames';
import { DotButton, DotIconButton, DotInputText } from '@digital-ai/dot-components';
import { HelpButton } from '../../../features/main-navigation/help-button/help-button';

import './textarea-markdown.component.less';

export interface TextareaMarkdownProps {
    inputId: string;
    isReadOnly?: boolean;
    maxRows?: number;
    name: string;
    onDelete?: (event: MouseEvent<HTMLButtonElement>) => void;
    onTextChange: (text: string) => void;
    rows: number;
    text: string;
}

export const TextAreaMarkdown = ({ inputId, isReadOnly = false, maxRows, name, onDelete, onTextChange, rows, text }: TextareaMarkdownProps) => {
    const [editing, setEditing] = useState<boolean>(false);
    const [originalText, setOriginalText] = useState<string>(text);
    const [currentText, setCurrentText] = useState<string>(text);

    const saveChanges = () => {
        onTextChange(currentText);
        setOriginalText(currentText);
        setEditing(false);
    };

    const revertChanges = () => {
        setCurrentText(originalText);
        setEditing(false);
    };

    const renderTextEditor = () => (
        <div>
            <DotInputText
                id={inputId}
                multiline={true}
                name={name}
                onChange={(e) => setCurrentText(e.target.value)}
                rows={rows}
                rowsMax={maxRows}
                value={currentText}
            />
            <div className="actions-wrapper">
                <DotButton data-testid="save-button" onClick={saveChanges} type="primary">
                    Save
                </DotButton>
                <DotButton data-testid="cancel-button" onClick={revertChanges} type="text">
                    Cancel
                </DotButton>
                <HelpButton helpLink="how-to/using-markdown-in-xl-release.html" />
            </div>
        </div>
    );

    const renderMarkDownPreview = () => (
        <div className={classNames('markdown-viewer', { 'read-only': isReadOnly })}>
            {!isReadOnly ? (
                <div className="markdown-viewer-actions">
                    {onDelete && <DotIconButton data-testid="delete-button" iconId="delete" onClick={onDelete} size="small" />}
                    <DotIconButton data-testid="edit-button" iconId="edit" onClick={() => setEditing(true)} size="small" />
                </div>
            ) : null}
            <ReactMarkdown className="markdown-wrapper">{currentText}</ReactMarkdown>
        </div>
    );

    return <div className="markdown-switcher">{editing ? renderTextEditor() : renderMarkDownPreview()}</div>;
};
