package com.xebialabs.xlrelease.support.cache.caffeine

import com.github.benmanes.caffeine.cache.{Cache, CacheLoader, Caffeine}
import com.typesafe.config.Config
import com.xebialabs.xlrelease.support.cache.config.CacheSettings
import com.xebialabs.xlrelease.support.config.TypesafeConfigExt.ExtendedConfig

import java.lang.management.ManagementFactory
import java.util.concurrent.TimeUnit
import javax.management.{ObjectInstance, ObjectName}

trait CaffeineCacheBuilder {
  def buildCache[K, V](cacheManagerName: String, name: String, cacheSettings: CacheSettings): Cache[K, V] =
    buildCache(cacheManagerName, name, cacheSettings, None)

  def buildCache[K, V](cacheManagerName: String, name: String, cacheSettings: CacheSettings,
                       loaderFn: Option[CacheLoader[K, V]]): Cache[K, V] = {

    val cacheConfig = cacheSettings.configForCache(name)
    val builder = configureCacheBuilder(cacheConfig)

    val invalidateByRegex = cacheConfig.getOptionalBoolean("invalidateWithRegex").getOrElse(false)
    val enabled = cacheSettings.isEnabled(name)
    val cache = loaderFn match {
      case Some(loader) =>
        val cache = builder.get.build[K, V](loader)
        new LoadingCaffeineCacheDelegate[K, V](name, cache, invalidateByRegex, enabled, loader)
      case None =>
        val cache = builder.get.build[K, V]()
        new CaffeineCacheDelegate[K, V](name, cache, invalidateByRegex, enabled)
    }

    if (cacheConfig.getOptionalBoolean("register-mbean").getOrElse(false)) {
      // D-27883: cache can be registered as mbean only if CaffeineCacheManager is not dynamic
      // (i.e. if it registers caches upfront) that way cache initialization will not be started from a script
      registerAsMbean(cacheManagerName, name, cache)
    }

    cache
  }

  private def configureCacheBuilder(cacheConfig: Config) = {
    val builder = for {
      b <- Some(Caffeine.newBuilder())
      b <- Some(cacheConfig.getOptionalInt("initial-capacity").fold(b)(s => b.initialCapacity(s)))
      b <- Some(cacheConfig.getOptionalLong("max-size").fold(b)(s => b.maximumSize(s)))
      b <- Some(cacheConfig.getOptionalLong("max-weight").fold(b)(s => b.maximumWeight(s)))
      b <- Some(cacheConfig.getOptionalBoolean("weak-keys").fold(b)(s => if (s) b.weakKeys() else b))
      b <- Some(cacheConfig.getOptionalBoolean("weak-values").fold(b)(s => if (s) b.weakValues() else b))
      b <- Some(cacheConfig.getOptionalBoolean("soft-values").fold(b)(s => if (s) b.softValues() else b))
      // ttl is same as expireAfterAccess (i.e. an alias)
      b <- Some(cacheConfig.getOptionalDuration("ttl").fold(b)(d => b.expireAfterAccess(d.toSeconds, TimeUnit.SECONDS)))
      b <- Some(cacheConfig.getOptionalDuration("expire-after-access").fold(b)(d => b.expireAfterAccess(d.toSeconds, TimeUnit.SECONDS)))
      b <- Some(cacheConfig.getOptionalDuration("expire-after-write").fold(b)(d => b.expireAfterWrite(d.toSeconds, TimeUnit.SECONDS)))
      b <- Some(cacheConfig.getOptionalDuration("refresh-after-write").fold(b)(d => b.refreshAfterWrite(d.toSeconds, TimeUnit.SECONDS)))
      b <- Some(cacheConfig.getOptionalBoolean("record-stats").fold(b)(s => if (s) b.recordStats() else b))
    } yield b
    builder
  }

  private lazy val server = ManagementFactory.getPlatformMBeanServer

  private def registerAsMbean[K, V](cacheManagerName: String, name: String, cache: Cache[K, V]): ObjectInstance = {
    val beanType = "statistics"
    val mbeanName = new ObjectName(s"com.xebialabs.xlrelease.cache:type=$beanType,CacheManager=$cacheManagerName,Cache=$name")
    val statistics = new CaffeineStatsCounter(cache)
    server.registerMBean(statistics, mbeanName)
  }
}
