package com.xebialabs.xlrelease.assertions;

import com.xebialabs.xlrelease.domain.Task;
import com.xebialabs.xlrelease.domain.status.TaskStatus;
import org.assertj.core.api.Assertions;
import org.joda.time.DateTime;

import java.util.List;

import static com.xebialabs.xlrelease.domain.status.TaskStatus.*;

public class TaskAssertion extends PlanItemAssertion<TaskAssertion, Task> {

    protected TaskAssertion(Task actual) {
        super(actual, TaskAssertion.class);
    }

    public TaskAssertion hasStatus(TaskStatus status) {
        isNotNull();
        Assertions.assertThat(actual.getStatus()).isEqualTo(status);
        return this;
    }

    public TaskAssertion hasId(String id) {
        isNotNull();
        Assertions.assertThat(actual.getId()).isEqualTo(id);
        return this;
    }

    public TaskAssertion isPlanned() {
        isNotNull().hasStatus(PLANNED);
        return this;
    }

    public TaskAssertion isInProgress() {
        isNotNull().hasStatus(IN_PROGRESS).hasStartDate();
        return this;
    }

    public TaskAssertion isSkipped() {
        isNotNull().hasStatus(SKIPPED);
        return this;
    }

    public TaskAssertion isFailing() {
        isNotNull().hasStatus(FAILING);
        return this;
    }

    public TaskAssertion isDone() {
        Assertions.assertThat(actual.getStatus().isDone()).isTrue();
        isNotNull().hasEndDate();
        return this;
    }

    public TaskAssertion isFailed() {
        isNotNull().hasStatus(FAILED);
        return this;
    }

    public TaskAssertion hasBeenFailed(int expectedFailuresCount) {
        isNotNull();
        Assertions.assertThat(actual.getFailuresCount()).isEqualTo(expectedFailuresCount);
        return this;
    }

    public TaskAssertion hasBeenFlagged() {
        isNotNull();
        Assertions.assertThat(actual.hasBeenFlagged()).isTrue();
        return this;
    }

    public TaskAssertion hasNotBeenFlagged() {
        isNotNull();
        Assertions.assertThat(actual.hasBeenFlagged()).isFalse();
        return this;
    }

    public TaskAssertion hasBeenDelayed() {
        isNotNull();
        Assertions.assertThat(actual.hasBeenDelayed()).isTrue();
        return this;
    }

    public TaskAssertion hasNotBeenDelayed() {
        isNotNull();
        Assertions.assertThat(actual.hasBeenDelayed()).isFalse();
        return this;
    }

    public TaskAssertion isCompleted() {
        isNotNull().hasStatus(COMPLETED);
        isNotNull().hasEndDate();
        return this;
    }

    public TaskAssertion isCompletedInAdvance() {
        isNotNull().hasStatus(COMPLETED_IN_ADVANCE);
        isNotNull().hasEndDate();
        return this;
    }

    public TaskAssertion isSkippedInAdvance() {
        isNotNull().hasStatus(SKIPPED_IN_ADVANCE);
        isNotNull().hasEndDate();
        return this;
    }

    public TaskAssertion isAborted() {
        isNotNull().hasStatus(ABORTED);
        isNotNull().hasEndDate();
        return this;
    }

    public TaskAssertion isPending() {
        isNotNull().hasStatus(PENDING);
        return this;
    }

    public TaskAssertion isPreconditionInProgress() {
        isNotNull().hasStatus(PRECONDITION_IN_PROGRESS);
        return this;
    }

    public TaskAssertion isWaitingForInput() {
        isNotNull().hasStatus(WAITING_FOR_INPUT);
        return this;
    }

    public TaskAssertion hasOwner(String owner) {
        isNotNull();
        Assertions.assertThat(actual.getOwner()).isEqualTo(owner);
        return this;
    }

    public TaskAssertion hasTitle(String title) {
        isNotNull();
        Assertions.assertThat(actual.getTitle()).isEqualTo(title);
        return this;
    }

    public TaskAssertion hasTags(List<String> tags) {
        isNotNull();
        Assertions.assertThat(actual.getTags()).containsAll(tags);
        return this;
    }

    public TaskAssertion hasDescription(String description) {
        isNotNull();
        Assertions.assertThat(actual.getDescription()).isEqualTo(description);
        return this;
    }

    public TaskAssertion hasTeam(String team) {
        isNotNull();
        Assertions.assertThat(actual.getTeam()).isEqualTo(team);
        return this;
    }

    public TaskAssertion hasIdContaining(String text) {
        isNotNull();
        Assertions.assertThat(actual.getId()).isNotNull().contains(text);
        return this;
    }

    public TaskAssertion isWaitingForScheduledStartDate() {
        isNotNull();
        Assertions.assertThat(actual.isWaitForScheduledStartDate()).isTrue();
        return this;
    }

    public TaskAssertion isDelayedDuringBlackout() {
        isNotNull();
        Assertions.assertThat(actual.isDelayDuringBlackout()).isTrue();
        return this;
    }

    public TaskAssertion hasComment(String commentId) {
        isNotNull();
        Assertions.assertThat(actual.findComment(commentId)).isNotNull();
        return this;
    }

    public TaskAssertion hasUpdatedModifiedAt(DateTime origModifiedAt) {
        isNotNull();
        Assertions.assertThat(actual.get$ciAttributes().getLastModifiedAt()).isNotNull();
        Assertions.assertThat(actual.get$ciAttributes().getLastModifiedAt()).isGreaterThan(origModifiedAt);
        return this;
    }
}
