package com.xebialabs.xlrelease.builder;

import java.util.*;

import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.xlrelease.domain.PollType;
import com.xebialabs.xlrelease.domain.Release;
import com.xebialabs.xlrelease.domain.ReleaseTrigger;
import com.xebialabs.xlrelease.domain.folder.Folder;
import com.xebialabs.xlrelease.domain.status.TriggerExecutionStatus;
import com.xebialabs.xlrelease.domain.variables.Variable;
import com.xebialabs.xlrelease.repository.Ids;

import static com.google.common.base.Strings.isNullOrEmpty;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static java.util.Map.Entry;
import static java.util.Objects.nonNull;

public class ReleaseTriggerBuilder {
    private String type;
    private String id;
    private Integer ciUid;
    private String folderId = "";
    private String title;
    private String description;
    private List<Variable> vars = newArrayList();
    private Release template;
    private String templateId;
    private String executionId;
    private boolean enabled = true;
    private boolean misfireRecoveryEnabled = false;
    private String releaseTitle;
    private String triggerState;
    private PollType pollType = PollType.REPEAT;
    private String periodicity = "60";
    private List<String> tags = newArrayList();
    private Map<String, Object> properties = newHashMap();
    private boolean initialFire;
    private Date lastRunDate;
    private TriggerExecutionStatus lastRunStatus;
    private String releaseFolder;

    private ReleaseTriggerBuilder(String type) {
        this.type = type;
    }

    public static ReleaseTriggerBuilder newReleaseTrigger(String type) {
        return new ReleaseTriggerBuilder(type);
    }

    public ReleaseTriggerBuilder withId(String id) {
        this.id = id;
        return this;
    }

    public ReleaseTriggerBuilder withCiUid(final Integer ciUid) {
        this.ciUid = ciUid;
        return this;
    }

    public ReleaseTriggerBuilder withTitle(String title) {
        this.title = title;
        return this;
    }

    public ReleaseTriggerBuilder withDescription(String description) {
        this.description = description;
        return this;
    }

    public ReleaseTriggerBuilder withIdAndTitle(String id) {
        this.id = id;
        this.title = id;
        return this;
    }

    public ReleaseTriggerBuilder withTemplateVariableValues(Map<String, String> values) {
        for (Entry<String, String> var : values.entrySet()) {
            vars.add(VariableBuilder.newStringVariable(var.getKey(), var.getValue()).build());
        }
        return this;
    }

    public ReleaseTriggerBuilder withTemplatePasswordVariableValues(Map<String, String> values) {
        for (Entry<String, String> var : values.entrySet()) {
            vars.add(VariableBuilder.newPasswordStringVariable(var.getKey(), var.getValue()).build());
        }
        return this;
    }

    public ReleaseTriggerBuilder withTemplateVariables(Variable... variables) {
        vars.addAll(Arrays.asList(variables));
        return this;
    }

    public ReleaseTriggerBuilder withTemplateVariables(List<Variable> variables) {
        vars.addAll(variables);
        return this;
    }

    public ReleaseTriggerBuilder withTemplate(Release template) {
        this.template = template;
        this.folderId = Ids.findFolderId(template.getId());
        return this;
    }

    public ReleaseTriggerBuilder withTemplateId(String templateId) {
        this.templateId = templateId;
        this.folderId = Ids.findFolderId(templateId);
        return this;
    }

    public ReleaseTriggerBuilder withFolder(Folder folder) {
        this.folderId = folder.getId();
        return this;
    }

    public ReleaseTriggerBuilder withFolderId(String folderId) {
        this.folderId = folderId;
        return this;
    }

    public ReleaseTriggerBuilder withExecutionId() {
        this.executionId = UUID.randomUUID().toString();
        return this;
    }

    public ReleaseTriggerBuilder withExecutionId(String id) {
        this.executionId = id;
        return this;
    }

    public ReleaseTriggerBuilder withNoExecutionId() {
        this.executionId = null;
        return this;
    }

    public ReleaseTriggerBuilder withEnabled(boolean enabled) {
        this.enabled = enabled;
        return this;
    }

    public ReleaseTriggerBuilder withMisfireRecoveryEnabled(boolean enabled) {
        this.misfireRecoveryEnabled = enabled;
        return this;
    }

    public ReleaseTriggerBuilder withReleaseTitle(String s) {
        this.releaseTitle = s;
        return this;
    }

    public ReleaseTriggerBuilder withTriggerState(String triggerState) {
        this.triggerState = triggerState;
        return this;
    }

    public ReleaseTriggerBuilder withInitialFire(boolean initialFire) {
        this.initialFire = initialFire;
        return this;
    }

    public ReleaseTriggerBuilder withPollType(PollType pollType) {
        this.pollType = pollType;
        return this;
    }

    public ReleaseTriggerBuilder withPeriodicity(String periodicity) {
        this.periodicity = periodicity;
        return this;
    }

    public ReleaseTriggerBuilder withProperty(String key, Object value) {
        this.properties.put(key, value);
        return this;
    }

    public ReleaseTriggerBuilder withTags(String... tags) {
        this.tags = newArrayList(tags);
        return this;
    }

    public ReleaseTriggerBuilder withLastRunDate(Date lastRunDate) {
        this.lastRunDate = lastRunDate;
        return this;
    }

    public ReleaseTriggerBuilder withLastRunStatus(TriggerExecutionStatus lastRunStatus) {
        this.lastRunStatus = lastRunStatus;
        return this;
    }

    public ReleaseTriggerBuilder withReleaseFolder(String releaseFolder) {
        this.releaseFolder = releaseFolder;
        return this;
    }

    public ReleaseTrigger build() {
        Type triggerType = Type.valueOf(this.type);
        ReleaseTrigger releaseTrigger = triggerType.getDescriptor().newInstance(id);

        releaseTrigger.setCiUid(ciUid);
        releaseTrigger.setTitle(title);
        releaseTrigger.setDescription(description);
        releaseTrigger.setVariables(vars);
        if (nonNull(template)) {
            releaseTrigger.setTemplate(template.getId());
        } else if (!isNullOrEmpty(templateId)) {
            releaseTrigger.setTemplate(templateId);
        }
        if (!isNullOrEmpty(releaseTrigger.getTemplate())) {
            // DO NOT Invoke this line : template.getReleaseTriggers().add(releaseTrigger)
            String templateFolderId = Ids.findFolderId(releaseTrigger.getTemplate());
            if (!templateFolderId.equals(folderId)) {
                throw new IllegalStateException("Trigger folder id does not match template folder id");
            }
        }
        releaseTrigger.setExecutionId(executionId);
        releaseTrigger.setEnabled(enabled);
        releaseTrigger.setMisfireRecoveryEnabled(misfireRecoveryEnabled);
        releaseTrigger.setReleaseTitle(releaseTitle);
        releaseTrigger.setTriggerState(triggerState);
        releaseTrigger.setInitialFire(initialFire);
        releaseTrigger.setPollType(pollType);
        releaseTrigger.setPeriodicity(periodicity);
        releaseTrigger.setFolderId(folderId);
        releaseTrigger.setLastRunDate(lastRunDate);
        releaseTrigger.setLastRunStatus(lastRunStatus);
        releaseTrigger.setReleaseFolder(releaseFolder);
        for (Entry<String, Object> entry : properties.entrySet()) {
            releaseTrigger.setProperty(entry.getKey(), entry.getValue());
        }
        releaseTrigger.setTags(tags);

        return releaseTrigger;
    }
}
