package com.xebialabs.xlrelease.serialization.json.utils

import com.fasterxml.jackson.databind.JsonNode
import com.fasterxml.jackson.databind.node.{ObjectNode, POJONode}
import com.fasterxml.jackson.databind.util.RawValue
import com.xebialabs.deployit.core.xml.PasswordEncryptingCiConverter
import com.xebialabs.deployit.plugin.api.services.Repository
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.util.PasswordEncrypter
import com.xebialabs.xlrelease.serialization.json.xltype.{CiJson2Reader, CiJson2Writer, XlrPasswordEncryptingCiConverter, XlrPasswordMaskingCiConverter}
import com.xebialabs.xltype.serialization.{CiWriter, ConfigurationItemConverter}

import java.io.{ByteArrayOutputStream, OutputStreamWriter, PrintWriter}
import java.nio.charset.StandardCharsets
import java.util.{List => JList}
import scala.util.Using

object CiSerializerHelper {

  import scala.jdk.CollectionConverters._

  private def nodeToString(jsonNode: JsonNode) = jsonNode match {
    case node: POJONode => node.getPojo match {
      case rawValue: RawValue => rawValue.rawValue().toString
      case other => other.toString
    }
    case _@node => node.toString
  }

  def newEncryptingConverter(): ConfigurationItemConverter = new XlrPasswordEncryptingCiConverter(PasswordEncrypter.getInstance())

  def deserialize(node: JsonNode, repository: Repository): ConfigurationItem = {
    deserialize(node, repository, newEncryptingConverter())
  }

  def deserialize(json: String, repository: Repository): ConfigurationItem = {
    deserialize(json, repository, newEncryptingConverter())
  }

  def deserialize(node: JsonNode, repository: Repository, ciConverter: ConfigurationItemConverter): ConfigurationItem = {
    removeNullFields(node)
    deserialize(nodeToString(node), repository, ciConverter)
  }

  def deserialize(json: String, repository: Repository, ciConverter: ConfigurationItemConverter): ConfigurationItem = {
    val ciReader = CiJson2Reader.create(json)
    val ci: ConfigurationItem = ciConverter.readCi(ciReader)
    if (repository != null) {
      ciConverter.resolveReferences(repository)
    }
    ci
  }

  private def removeNullFields(node: JsonNode): Unit = {
    node match {
      case objectNode: ObjectNode =>
        val nullFields = objectNode.fields().asScala.filter(_.getValue.isNull).map(_.getKey).toSeq.asJava
        objectNode.remove(nullFields)
        objectNode.fields().asScala
          .filter(_.getValue.isContainerNode)
          .foreach { field => removeNullFields(field.getValue) }
      case _ =>
    }
  }

  def serialize(ci: ConfigurationItem): String = serialize(ci, new PasswordEncryptingCiConverter())

  def serializeWithMaskedPasswords(ci: ConfigurationItem): String = serialize(ci, new XlrPasswordMaskingCiConverter())

  def serialize(cis: JList[_ <: ConfigurationItem]): String = serialize(cis, new PasswordEncryptingCiConverter())

  def serialize(ci: ConfigurationItem, converter: ConfigurationItemConverter): String =
    convert(converter) { (ciConverter, ciWriter) => ciConverter.writeCi(ci, ciWriter, Integer.MAX_VALUE) }

  def serialize(cis: JList[_ <: ConfigurationItem], converter: ConfigurationItemConverter): String =
    convert(converter) { (ciConverter, ciWriter) => ciConverter.writeCis(cis.asInstanceOf[JList[ConfigurationItem]], ciWriter, Integer.MAX_VALUE) }

  private def convert(converter: ConfigurationItemConverter)(action: (ConfigurationItemConverter, CiWriter) => Unit): String = {
    val byteArrayOutStream = new ByteArrayOutputStream()
    val outputStreamWriter = new OutputStreamWriter(byteArrayOutStream, StandardCharsets.UTF_8)
    val printWriter: PrintWriter = new PrintWriter(outputStreamWriter, true)
    Using.resource(new CiJson2Writer(printWriter)) { ciWriter =>
      action(converter, ciWriter)
    }
    printWriter.flush()
    byteArrayOutStream.toString(StandardCharsets.UTF_8.name())
  }

}
