package com.xebialabs.xlrelease.serialization.json.xltype;

import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import jakarta.json.*;

import com.xebialabs.xlrelease.serialization.json.utils.JsonWithCachedProvider;
import com.xebialabs.xltype.serialization.CiListReader;
import com.xebialabs.xltype.serialization.CiReader;
import com.xebialabs.xltype.serialization.SerializationException;

import static jakarta.json.JsonValue.ValueType.OBJECT;

public class CiListJson2Reader implements CiListReader {

    private final Iterator<JsonObject> iterator;
    private JsonObject currentObject;

    public CiListJson2Reader(JsonArray json) {
        try {
            this.iterator = toObjectList(json).iterator();
        } catch (JsonException e) {
            throw new SerializationException(e);
        }
    }

    public static CiListReader create(String jsonArray) {
        try (Reader reader = new StringReader(jsonArray);
             JsonReader jsonReader = JsonWithCachedProvider.createReader(reader)) {
            return new CiListJson2Reader(jsonReader.readArray());
        } catch (IOException | JsonException e) {
            throw new IllegalArgumentException("Can't parse the following as a JSON array:\n" + jsonArray, e);
        }
    }

    //
    // CiListReader implementation
    //

    @Override
    public boolean hasMoreChildren() {
        return iterator.hasNext();
    }

    @Override
    public void moveIntoChild() {
        currentObject = iterator.next();
    }

    @Override
    public void moveOutOfChild() {
        // Noop
    }

    @Override
    public CiReader getCurrentCiReader() {
        CiReader currentReader = new CiJson2Reader(currentObject);

        // Unwrap single value, so it can be read by calling CiReader.getStringValue()
        // (Hack to support 'streaming reading' as mandated by the XStream implementation)
        if (currentObject.containsKey("ci-ref")) {
            currentReader.moveIntoProperty();
        }

        return currentReader;
    }


    //
    // Util
    //

    private static List<JsonObject> toObjectList(JsonArray array) {
        List<JsonObject> objects = new ArrayList<>(array.size());
        for (int i = 0; i < array.size(); i++) {
            JsonValue jsonValue = array.get(i);
            JsonObject jsonObject = null;
            if (jsonValue.getValueType() == OBJECT) {
                jsonObject = array.getJsonObject(i);
            }
            // Wrap single value into JsonObject
            if (jsonObject == null) {
                // We use a dash in the property name so it won't clash with CI properties.
                jsonObject = JsonWithCachedProvider.createObjectBuilder().add("ci-ref", array.getJsonString(i)).build();
            }
            objects.add(jsonObject);
        }

        return objects;
    }

}
